
#ifndef __VOICEMAN_LANG_H__
#define __VOICEMAN_LANG_H__

/**\brief The abstract class to unify all language processing objects*/
class Lang
{
public:
  enum CharType {UPCASE=0, LOWCASE=1, OTHER=2};

  /**\brief The destructor*/
  virtual ~Lang() {}

  /**\brief Returns type of specified character in language of current object
   *
   * \param [in] c The character to get type of
   */
  virtual CharType getCharType(wchar_t c) const = 0;

  /**\brief Checks if specified characters represent same letter
   *
   * Almost in every language each letter can be represented in different
   * cases and different characters are used to store them. This method
   * checks, if specified characters represent the same letter.
   * \param [in] c1 The first character to compare
   * \param [in] c2 The second character to compare
   */
  virtual bool equalChars(wchar_t c1, wchar_t c2) const = 0;

  /**\brief Replaces numbers with their word representation
   *
   * \param [in/out] str The string to process
   * \param [in] singleDigits Process each digit as separate number
   */
  virtual void expandNumbers(std::wstring& str, bool singleDigits) const = 0;

  /**\brief Returns all characters of this language alphabet (in all cases)*/
  virtual std::wstring getAllChars() const = 0;

  /**\brief Returns upper case of the specified character
   *
   * \param [in] ch The character to return upper case of
   */
  virtual wchar_t toUpper(wchar_t ch) const = 0;

  /**\brief returns lower case of the specified character
   *
   * \param [in] ch The character to return lower case of
   */
  virtual wchar_t toLower(wchar_t ch) const = 0;

  /**\brief Returns string with letters adjusted to upper case
   *
   * \param [in] str The string to adjust case
   */
  virtual std::wstring toUpper(const std::wstring& str) const = 0;

  /**\brief Returns string with letters adjusted to lower case
   *
   * \param [in] str The string to adjust case
   */
  virtual std::wstring toLower(const std::wstring& str) const = 0;

  /**\brief Inserts additional spaces to better words processing
   *
   * This method inserts additional spaces to split words to avoid using
   * capital letters in the middle of the words. If capital letter used in
   * the middle of the word this method will insert space before it. It is
   * useful for Hungarian notation.
   *
   * \param [in] text The string to process
   */
  virtual std::wstring separate(const std::wstring& text) const = 0;

  /**\brief Marks letters to speak separately (usual acronims)
   *
   * \param [in] text The string to process
   * \param [out] marks The boolean vector to save marks
   */
  virtual void markCapitals(const std::wstring& text, BoolVector& marks) const = 0;
};//class Lang;

typedef std::vector<const Lang*> ConstLangPtrVector;

/**\brief The abstract interface for all classes to get language objects by language identifier
 *
 * In daemon code language objects are usually identified by special
 * language ID values . This interface declares method to translate such
 * language identifiers to a valid language objects for further
 * operations.
 */
class AbstractLangIdResolver
{
public:
  /**\brief The destructor*/
  virtual ~AbstractLangIdResolver() {}

  /**\brief Returns language object by language ID
   *
   * This method checks if provided language identifier is valid and
   * returns a pointer to corresponding language object or NULL otherwise.
   *
   * \param [in] langId The language identifier to get object by
   *
   * \return The pointer to language object or NULL if language ID is unknown
   */
  virtual const Lang* getLangById(LangId langId) const = 0;
}; //class AbstractLangIdResolver;

#endif //__VOICEMAN_LANG_H__
