//FIXME:good comments;
#ifndef __VOICEMAN_OUTPUT_H__
#define __VOICEMAN_OUTPUT_H__

#include"TextItem.h"
#include"Lang.h"

/**\brief The class with complete information about valid and ready to use output
 *
 * The output in VoiceMan terminology means one available speech
 * synthesizer with all its corresponding settings, required to invoke it
 * and make speech generation. One available in your system synthesizer
 * must not be represented exactly by one output, there can be several
 * outputs for each TTS witrh different parameters. This class does not
 * perform synthesizer execution, it only stores information about it and
 * makes some data preprocessing. External synthesizer calls are made by
 * separate process, called executor. It is managed by ExecutorInterface
 * class.
 */
class Output
{
public:
  /**\brief The constructor*/
  Output()
    : m_langId(LANG_ID_NONE), m_lang(NULL) {}

  /**\brief The destructor*/
  virtual ~Output() {}

  /**\brief Returns the language ID this output is processing*/
  LangId getLangId() const
  {
    return m_langId;
  }

  /**\brief Sets new language IDthis output is processing
   *
   * \param [in] langId The ID of the language to set
   */
  void setLangId(LangId langId)
  {
    m_langId = langId;
  }

  /**\brief Sets new pointer to language object
   *
   * Language object is used by Output class for miscellaneous text
   * processing before sending to executor. This method allows you get
   * current stored value of Lang pointer.
   *
   * \return The pointer to used language object
   */
  const Lang* getLang() const
  {
    return m_lang;
  }

  /**\brief Sets new pointer to language object
   *
   * Language object is used by Output class for miscellaneous text
   * processing before sending to executor. This method allows you set
   * current stored value of Lang pointer.
   *
   * \param [in] lang The pointer to language object to set
   */
  void setLang(const Lang* lang)
  {
    m_lang = lang;
  }

  /**\brief Returns the name of this output*/
  std::string getName() const
  {
    return m_name;
  }

  /**\brief Sets new name for this output
   *
   * \param [in] name The new name to set
   */
  void setName(const std::string& name)
  {
    m_name = name;
  }

  /**\brief Returns the voice family of this output*/
  std::string getFamily() const
  {
    return m_family;
  }

  /**\brief Sets new voice family of this output
   *
   * \param [in] family The new voice family to set
   */
  void setFamily(const std::string& family)
  {
    m_family = family;
  }

  /**\brief Generates the command line to execute speech synthesizer 
   *
   * \param [in] textItem The part of text data to generate command line for
   */
  std::string prepareSynthCommand(const TextItem& textItem)const;

  /**\brief Generates the command line to execute ALSA player
   *
   * \param [in] textItem The part of text data to generate command line for
   */
  std::string prepareAlsaPlayerCommand(const TextItem& textItem)const;

  /**\brief Generates the command line to execute PulseAudio player
   *
   * \param [in] textItem The part of text data to generate command line for
   */
  std::string preparePulseaudioPlayerCommand(const TextItem& textItem)const;

  /**\brief Generates the command line to execute PC speaker player
   *
   * \param [in] textItem The part of text data to generate command line for
   */
  std::string preparePcspeakerPlayerCommand(const TextItem& textItem)const;

  /**\brief Prepares text to send to speech synthesizer
   *
   * \param [in] textItem The text item to prepare text of
   */
  std::string prepareText(const TextItem& textItem) const;

  /**\brief Sets new command line template to run synthesizer
   *
   * \param [in] cmdLine The command line pattern to set
   */
  void setSynthCommand(const std::string& cmdLine)
  {
    m_synthCommand = cmdLine;
  }

  /**\brief Sets new command line template to run ALSA player
   *
   * \param [in] cmdLine The command line pattern to set
   */
  void setAlsaPlayerCommand(const std::string& cmdLine)
  {
    m_alsaPlayerCommand = cmdLine;
  }

  /**\brief Sets new command line template to run PulseAudio player
   *
   * \param [in] cmdLine The command line pattern to set
   */
  void setPulseaudioPlayerCommand(const std::string& cmdLine)
  {
    m_pulseaudioPlayerCommand = cmdLine;
  }

  /**\brief Sets new command line template to run PC speaker player
   *
   * \param [in] cmdLine The command line pattern to set
   */
  void setPcspeakerPlayerCommand(const std::string& cmdLine)
  {
    m_pcspeakerPlayerCommand = cmdLine;
  }

  /**\brief Adds new replacement to mark capitalized letter
   *
   * \param [in] c The letter being marked
   * \param [in] value The text string to replace with
   */
  void addCapMapItem(wchar_t c, const std::wstring& value);

  /**\brief Sets the format of pitch value used in command lines
   *
   * This value will be used in synthesizer and player calls during speech
   * generation. Purpose of each parameter is describe in TextParam
   * documentation. If different players require different form of some
   * parameter, you should avoid it specification in players command line
   * and use it only in command line of synthesizer.
   * 
   * \param [in] digits Number of digits after decimal dot 
   * \param [in] min The number to translate minimal parameter value into
   * \param [in] aver The number to translate average parameter value into
   * \param [in] max The number to translate maximum parameter value into
   */
  void setPitchFormat(size_t digits, double min, double aver, double max);

  /**\brief Sets the format of rate value used in command lines
   *
   * This value will be used in synthesizer and player calls during speech
   * generation. Purpose of each parameter is describe in TextParam
   * documentation. If different players require different form of some
   * parameter, you should avoid it specification in players command line
   * and use it only in command line of synthesizer.
   * 
   * \param [in] digits Number of digits after decimal dot 
   * \param [in] min The number to translate minimal parameter value into
   * \param [in] aver The number to translate average parameter value into
   * \param [in] max The number to translate maximum parameter value into
   */
  void setRateFormat(size_t digits, double min, double aver, double max);

  /**\brief Sets the format of volume value used in command lines
   *
   * This value will be used in synthesizer and player calls during speech
   * generation. Purpose of each parameter is describe in TextParam
   * documentation. If different players require different form of some
   * parameter, you should avoid it specification in players command line
   * and use it only in command line of synthesizer.
   * 
   * \param [in] digits Number of digits after decimal dot 
   * \param [in] min The number to translate minimal parameter value into
   * \param [in] aver The number to translate average parameter value into
   * \param [in] max The number to translate maximum parameter value into
   */
  void setVolumeFormat(size_t digits, double min, double aver, double max);

private:
  struct FloatValueFormat
  {
    FloatValueFormat()
      : digits(2), min(0), aver(0.5), max(1) {}

    size_t digits;
    double min;
    double aver;
    double max;
  }; //struct FloatValueFormat;

private:
  std::wstring makeCaps(const TextItem& textItem) const;
  std::string prepareCommandLine(const std::string& pattern, const TextItem& textItem) const;
  std::string prepareFloatValue(TextParam value, const FloatValueFormat& format) const;

private:
  typedef std::map<wchar_t, std::wstring> WCharToWStringMap;

  LangId m_langId;
  const Lang* m_lang;
  std::string m_name, m_family;
  WCharToWStringMap m_capList;
  std::string m_synthCommand;
  std::string m_alsaPlayerCommand, m_pulseaudioPlayerCommand, m_pcspeakerPlayerCommand;
  FloatValueFormat m_pitchFormat;
  FloatValueFormat m_rateFormat;
  FloatValueFormat m_volumeFormat;
}; //class Output;

typedef std::vector<Output> OutputVector;
typedef std::list<Output> OutputList;

#endif // __VOICEMAN_OUTPUT_H__
