
#include<assert.h>
#include<stdlib.h>
#include<stdio.h>
#include<string.h>
#include<sys/types.h>
#include<unistd.h>
#include<sys/stat.h>
#include<fcntl.h>
#include<sys/signal.h>
#include<sys/wait.h>
#include<signal.h>
#include<errno.h>
#include<locale.h>
#include"executorCommandHeader.h"

#define ERROR_PREFIX "voiceman-executor:"

#define NULL_DEVICE "/dev/null"
#define IO_BUF_SIZE 2048

void playTone(size_t freq, size_t lengthMs);

typedef struct QueueItem_  
{
  char* synthCommand;
  char* playerCommand;
  char* text;
  struct QueueItem_* next;
} QueueItem;

pid_t pid = 0;
pid_t playerPid = 0;
QueueItem* queueHead = NULL;
QueueItem* queueTail = NULL;
size_t maxQueueSize = 0;
sig_atomic_t wasSigChld = 0;

void sigChldHandler(int n)
{
  wasSigChld = 1;
}

/*Reads block of specified length and produces subsequent calls of read() in case of short read operation*/
ssize_t readBlock(int fd, void* buf, size_t bufSize)
{
  char* b = (char*)buf;
  size_t c = 0;
  assert(buf);
  while(c < bufSize)
    {
      ssize_t res = read(fd, &b[c], bufSize - c);
      if (res == -1)
	return -1;
      if (res == 0)
	break;
      assert(res > 0);
      c += (size_t)res;
    } /*while();*/
  return (ssize_t)c;
}

/*RWrites block of specified length and produces subsequent calls of write() in case of short write operation*/
ssize_t writeBlock(int fd, void* buf, size_t bufSize)
{
  char* b = (char*)buf;
  size_t c = 0;
  assert(buf);
  while(c < bufSize)
    {
      ssize_t res = write(fd, &b[c], bufSize - c);
      if (res == -1)
	return -1;
      assert(res >= 0);
      c += (size_t)res;
    } /*while();*/
  assert(c == bufSize);
  return (ssize_t)c;
}

/*Reads buffer of any length producing subsequent calls to read fixed length blocks*/
ssize_t readBuffer(int fd, void* buf, size_t bufSize)
{
  char* b = (char*)buf;
  size_t c = 0;
  assert(buf);
  while(c < bufSize)
    {
      size_t requiredSize = bufSize > c + IO_BUF_SIZE?IO_BUF_SIZE:(size_t)(bufSize - c);
      ssize_t res = readBlock(fd, &b[c], requiredSize);
      if (res == -1)
	return -1;
      c += (size_t)res;
      if (res < (ssize_t)requiredSize)
	break;
    } /*while();*/
  return (ssize_t)c;
}

/*Writes buffer of any length producing subsequent calls to write fixed length blocks*/
ssize_t writeBuffer(int fd, void* buf, size_t bufSize)
{
  char* b = (char*)buf;
  size_t c = 0;
  assert(buf);
while(c < bufSize)
    {
      size_t requiredSize = bufSize > c + IO_BUF_SIZE?IO_BUF_SIZE:(size_t)(bufSize - c);
      ssize_t res = writeBlock(fd, &b[c], requiredSize);
      if (res == -1)
	return -1;
      assert(res == (ssize_t)requiredSize);
      c += (size_t)res;
    } /*while();*/
  assert(c == bufSize);
  return (ssize_t)c;
}

void onNoMemError()
{
  fprintf(stderr, "%sno enough free memory\n", ERROR_PREFIX);
  exit(EXIT_FAILURE);
}

void onSystemCallError(const char* descr, int errorCode)
{
  assert(descr != NULL);
  fprintf(stderr, "%s%s:%s", ERROR_PREFIX, descr, strerror(errorCode));
  exit(EXIT_FAILURE);
}

void putNewItemToQueue(char* synthCommand, char* playerCommand, char* text)
{
  QueueItem* newItem;
  assert(synthCommand);
  assert(playerCommand);
  assert(text);
  newItem = (QueueItem*)malloc(sizeof(QueueItem));
  if (!newItem)
    onNoMemError();
  newItem->synthCommand = synthCommand;
  newItem->playerCommand = playerCommand;
  newItem->text = text;
  newItem->next = NULL;
  if (!queueHead)/*there are no items in queue at all*/
    {
      queueHead = newItem;
      queueTail = newItem;
      return;
    }
  assert(queueTail);
  queueTail->next = newItem;
  queueTail = newItem;
}

void popQueueFront()
{
  if (!queueHead)
    {
      assert(!queueTail);
      return;
    }
  free(queueHead->synthCommand);
  free(queueHead->playerCommand);
  free(queueHead->text);
  if (queueHead->next)/*it is not the single item in queue*/
    {
      QueueItem* p = queueHead;
      queueHead = queueHead->next;
      free(p);
      return;
    }
  assert(queueHead == queueTail);
  free(queueHead);
  queueHead = NULL;
  queueTail = NULL;
}

void eraseQueue()
{
  while(queueHead)
    popQueueFront();
}

void execute(char* synthCommand, char* playerCommand, char* text)
{
  int pp[2];
  int interPp[2];
  size_t textLen = strlen(text);
  ssize_t res;
  assert(synthCommand);
  assert(playerCommand);
  assert(text);
  assert(pid == (pid_t)0);
  assert(playerPid == (pid_t)0);
  if (pipe(pp) == -1)
    {
      perror("pipe()");
      return;
    }
  if (pipe(interPp) == -1)
    {
      perror("pipe()");
      close(pp[0]);
      close(pp[1]);
      return;
    }
  pid = fork();
  if (pid == (pid_t)-1)
    {
      perror("fork()");
      close(pp[0]);
      close(pp[1]);
      close(interPp[0]);
      close(interPp[1]);
      pid = 0;
      return;
    }
  if (pid == (pid_t)0)/*The child process*/
    {
      int fd = open(NULL_DEVICE, O_WRONLY);
      if (fd == -1)
	{
	  perror("open(NULL_DEVICE)");
	  exit(EXIT_FAILURE);
	}
      setpgrp();
      close(pp[1]);/*Closing pipe input end*/
      close(interPp[0]);/*Closing pipe output end*/
      dup2(pp[0], STDIN_FILENO);
      dup2(interPp[1], STDOUT_FILENO);
      dup2(fd, STDERR_FILENO);
      if (execlp("/bin/sh", "/bin/sh", "-c", synthCommand, NULL) == -1)
	{
	  perror("execlp(/bin/sh)");
	  exit(EXIT_FAILURE);
	}
    } /* child process*/

  playerPid = fork();
  if (playerPid == (pid_t)-1)
    {
      /*We cannot create new child process for player, closing all pipes and wait synth process*/
      close(pp[0]);
      close(pp[1]);
      close(interPp[0]);
      close(interPp[1]);
      waitpid(pid, NULL, 0);
      pid = 0;
      playerPid = 0;
      return;
    }
  if (playerPid == (pid_t)0)/*player child process*/
    {
      int fd;
      setpgid(0, pid);
      fd = open(NULL_DEVICE, O_WRONLY);
      if (fd == -1)
	{
	  perror("open(NULL_DEVICE)");
	  exit(EXIT_FAILURE);
	}
      close(pp[0]);
      close(pp[1]);
      close(interPp[1]);/*Closing pipe input end*/
      dup2(interPp[0], STDIN_FILENO);
      dup2(fd, STDOUT_FILENO);
      dup2(fd, STDERR_FILENO);
      if (execlp("/bin/sh", "/bin/sh", "-c", playerCommand, NULL) == -1)
	{
	  perror("execlp(/bin/sh)");
	  exit(EXIT_FAILURE);
	}
    } /*player child process*/
  close(interPp[0]);
  close(interPp[1]);
  close(pp[0]);/*Closing output side of pipe*/
  res = writeBuffer(pp[1], text, textLen);
  if (res < 0)
    {
      perror("write()");
      close(pp[1]);
      return;
    }
  assert(res == (ssize_t)textLen);
  if (writeBlock(pp[1], "\n", 1) == -1)
    perror("write()");
  close(pp[1]);
}

void playNext()
{
  if (queueHead == NULL)/*No more text items to play*/
    {
      /*we must notify, there are no more items to play*/
      printf("silence\n");
      fflush(stdout);
      return;
    }
  execute(queueHead->synthCommand, queueHead->playerCommand, queueHead->text);
  popQueueFront();
}

/*This function frees provided string buffers if necessary*/
void play(char* synthCommand, char* playerCommand, char* text)
{
  assert(synthCommand);
  assert(playerCommand);
  assert(text);
  if (pid != 0)/*playback in progress now*/
    {
      putNewItemToQueue(synthCommand, playerCommand, text);
      return;
    }
  execute(synthCommand, playerCommand, text);
  free(synthCommand);
  free(playerCommand);
  free(text);
}

void stop()
{
  eraseQueue();
  if (pid == (pid_t)0)/*There is no playback now*/
    return;
  if (playerPid != (pid_t)0)/*just for resistance, must not happen*/
    {
      kill(playerPid, SIGINT);
      kill(playerPid, SIGKILL);
    }
  killpg(pid, SIGKILL);
  while(waitpid(-1 * pid, NULL, 0) >= 0);
  pid = 0;
  playerPid = 0;
  printf("stopped\n");
  fflush(stdout);
}

/*This function returnes zero if there is no more data and handle can be closed*/
char processInputCommand(int fd)
{
  CommandHeader header;
  ssize_t res;
  res = readBlock(fd, &header, sizeof(header));
  if (res < 0)
    onSystemCallError("read()", errno);
  if (res < sizeof(CommandHeader))
    return 0;
  if (header.code == COMMAND_STOP)
    {
      stop();
      return 1;
    } /*COMMAND_STOP*/
  if (header.code == COMMAND_SAY)
    {
      char* synthCommand;
      char* playerCommand;
      char* text;
      synthCommand = (char*)malloc(header.param1);
      if (!synthCommand)
	onNoMemError();
      playerCommand = (char*)malloc(header.param2);
      if (!playerCommand)
	onNoMemError();
      text = (char*)malloc(header.param3);
      if (text == NULL)
	onNoMemError();
      /*reading synth command line*/
      res = readBuffer(fd, synthCommand, header.param1);
      if (res < 0)
	onSystemCallError("read()", errno);
      if (res < header.param1)
	{
	  free(synthCommand);
	  free(playerCommand);
	  free(text);
	  return 0;
	}
      /*reading player command line*/
      res = readBuffer(fd, playerCommand, header.param2);
      if (res < 0)
	onSystemCallError("read()", errno);
      if (res < header.param2)
	{
	  free(synthCommand);
	  free(playerCommand);
	  free(text);
	  return 0;
	}
      /*reading text to say*/
      res = readBuffer(fd, text, header.param3);
      if (res < 0)
	onSystemCallError("read()", errno);
      if (res < header.param3)
	{
	  free(synthCommand);
	  free(playerCommand);
	  free(text);
	  return 0;
	}
      play(synthCommand, playerCommand, text);
      return 1;
    } /*COMMAND_EXECUTE*/
  if (header.code == COMMAND_TONE)
    {
      /*FIXME:*/
      return 1;
    } /*COMMAND_TONE*/
  if (header.code == COMMAND_SET_QUEUE_LIMIT)
    {
      maxQueueSize = header.param1;
      return 1;
    } /*COMMAND_SET_QUEUE_LIMIT*/
  fprintf(stderr, "%s unknown command %X(%lu,%lu,%lu)\n", ERROR_PREFIX, header.code, header.param1, header.param2, header.param3);
  fflush(stderr);
  return 1;
}

void handleSignals()
{
  if (wasSigChld)
    {
      pid_t pp = 0;
      wasSigChld = 0;
      if (pid == (pid_t)0)
	{
	  /*there was not any playback before*/
	  /*SIGCHLD may be caused by SIGKILL signals to all children*/
	  playerPid = 0;/*in any case*/
	  return;/*we must not do anything*/
	}
      /*Analyzing player status*/
	  pp = waitpid(playerPid, NULL, WNOHANG);
      while(1)
	{
	  pid_t p = waitpid(-1 * pid, NULL, WNOHANG);
	  if (p == (pid_t)0)
	    return;/*there are some live processes, we must wait new SIGCHLD to pick them*/
	  if (p < (pid_t)0)
	    break;
	  /*yes, we have picked up real zombie and must try new waitpid(), there can be more*/
	}
      /*
       * "pp" is the status of player process. If "pp" variable is zero it
       * means player process is still alive. In original idea this point
       * cannot be reached if one of the processes in main process group is
       * still running, but there are very and very important thing: the player
       * process is not joined into process group exactly at the moment of its
       * creation and there is tiny delay during which it is not in group. So,
       * we must do special checking to be sure the player process is not
       * alive.
       */
      if (pp == 0)
	return;
      pid = 0;
      playerPid = 0;
      playNext();
    }
}

int mainLoop(int fd, sigset_t* sigMask)
{
  /*Endless loop for all commands*/
  while(1)
    {
      /*Preparing for pselect() call*/
      fd_set fds;
      FD_ZERO(&fds);
      FD_SET(fd, &fds);
      /*Calling pselect()*/
      if (pselect(fd + 1, &fds, NULL, NULL, NULL, sigMask) == -1)
	{
	  /*pselect() has returned -1, what the problem*/
	  int errorCode = errno;
	  if (errorCode == EINTR)
	    {
	      /*Now we must handle every signal we caught*/
	      handleSignals();
	      /*Trying to call pselect() one more time*/
	      continue;
	    }
	  /*it is an unexpected system call error, we must stop processing*/
	  onSystemCallError("pselect()", errorCode);
	} /*if (pselect() == -1)*/
      assert(FD_ISSET(fd, &fds));
      if (!processInputCommand(fd))/*this function returnes zero if fd was closed*/
	return 0;
    } /*while(1)*/
}

int main(int argc, char* argv[])
{
  struct sigaction sa;
  sigset_t origMask, blockedMask;
  setlocale(LC_ALL, "");
  /*Installing SIGCHLD signal handler*/
  sigaction(SIGCHLD, NULL, &sa);
  sa.sa_handler = sigChldHandler;
  sa.sa_flags |= SA_RESTART;
  sigaction(SIGCHLD, &sa, NULL);
  /*Preparing blocking signal mask with disabled SIGCHLD*/
  sigemptyset(&blockedMask);
  sigaddset(&blockedMask, SIGCHLD);
  sigprocmask(SIG_BLOCK, &blockedMask, &origMask);
  return mainLoop(STDIN_FILENO, &origMask);
}
