
#ifndef __VOICEMAN_CMD_ARGS_PARSER_H__
#define __VOICEMAN_CMD_ARGS_PARSER_H__

/**\brief Represents information about available command line argument
 *
 * This structure is used to define a static array and declare all
 * available command line arguments for application. Information from
 * this array used to parse command line arguments provided by user and
 * print help message.
 */
struct CMDARG
{
  /**\brief Single-letter argument name (space means no more arguments in table)*/
  char shortName;

  /**\brief Single-word argument name*/
  const char* longName;

  /**\brief Additional parameter name if this argument has any (empty string means no parameter)*/
  const char* param;

  /**\brief The string to comment this argument in help screen*/
  const char*descr;
}; //struct CMDARG;

/**\brief Parses command line arguments and prints help screen
 *
 * This class receives command line arguments and compares them with the
 * table specified in application source code to determine invalid usage
 * and make easy checks of their presence. Also this class can print
 * prepared help screen with all available arguments and their
 * description.
 */
class CmdArgsParser
{
public:
  /**\brief The constructor
   *
   * Number of all available arguments is determined implicitly by the
   * items of allParam argument. The last item in this array must have
   * space in its shortName field.
   * \param [in] allParams The array with available command line arguments
   */
  CmdArgsParser(CMDARG* allParams);

  /**\brief Prints description of all available arguments*/
  void printHelp() const;

  /**\brief Checks if specified parameter was used at program startup
   *
   * \param [in] name The parameter to check specified by longName field value
   */
  bool used(const std::string& name) const;

  /**\brief Returns additional argument value of command line parameter
   *
   * \param [in] The name of parameter to get argument of, specified by longName field value
   */
  const std::string operator [](const std::string& name) const;

  /**\brief Performs parsing of arguments provided by user
   *
   * This method performs complete command line parsing and fills all
   * internal structures to save the result. If there are invalid arguments
   * specified by user this method prints error description and returns
   * zero.
   *
   * \param [in] argc The argc parameter received from main() function call
   * \param [in] argv The argv parameter received from main() function call
   *
   * \return Non-zero if there are no errors or zero if there are invalid command line arguments
   */
  bool parse(int argc, char* argv[]);

public:
  /**\brief The set of arguments specified after known command line parameters*/
  const std::vector<std::string>& files;

private:
  struct USEDPARAM
  {
    USEDPARAM() 
      : hasValue(0) {}

    std::string name, value;
    bool hasValue;
  }; //struct USEDPARAM;

  typedef std::vector<USEDPARAM> UsedParamVector;

private:
  size_t identifyParam(char* p) const;

private:
  typedef std::vector<std::string> StringVector;

  StringVector m_files;
    UsedParamVector m_usedParams;
    CMDARG* m_availableParams;
  size_t m_availableParamCount;
}; // class Cmdargs;

#endif //__VOICEMAN_CMD_ARGS_PARSER_H__
