/*
 * Copyright (C) 2007 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <cassert>	// for assert()
#include <cstdlib>	// for exit() and EXIT_FAILURE constant
#include <fstream>	// for ifstream
#include <utility>	// for make_pair()

#include "LineOfSourceCode.hh"	// for LineOfSourceFile and SourceFile classes

using std::ifstream;
using std::ostream;
using std::string;
using std::list;
using std::multimap;
using std::cerr;
using std::cout;
using std::endl;

SourceFile::SourceFile(const char *filename, bool rflag, bool pheader)
				: fileName(filename), recursive_mode(rflag), printHeader(pheader), fileWasOpened(true) {

	string line;
	size_t lineno = 1;
	string filePath;	// used only when -r flag specified

	// We can't use dirname() here because Windows doesn't have her
	if (rflag) {
#ifndef _WIN32
		string::size_type pos = fileName.find_last_of('/');
#else
		string::size_type pos = fileName.find_last_of('\\');
#endif
		if (pos != string::npos){
			filePath = fileName.substr(0, pos+1);
		}
	}

	ifstream file(filename);
	if (!file) {
		if (printHeader) {
			cerr << "Can't open file " << filename << endl;
			exit(EXIT_FAILURE);
		} else {
			fileWasOpened = false;
			return;
		}
	}

	while (getline(file, line)) {
		LineOfSourceCode *losc = new LineOfSourceCode(line, lineno++);

		if (printHeader && (losc->isSystemInclude() || losc->isUserInclude())) {
			addHeader(losc);
		}


		if (rflag && losc->isUserInclude()) {

			string path = filePath + losc->getHeaderName();

			SourceFile *usf = new SourceFile(path.c_str(), true, false);
			losc->usf = usf;
		}

		sourceFile.push_back(losc);
	}

	file.close();
}

SourceFile::~SourceFile() {

	for (list<LineOfSourceCode *>::const_iterator lst = sourceFile.begin();
			lst != sourceFile.end();
			++lst) {
		delete *lst;
	}
}

void
SourceFile::addHeader(LineOfSourceCode *losc) {
	hdb.insert(make_pair(losc->getHeaderName(), losc));

}

void
SourceFile::printDuplicateHeaders(const string &name, size_t lineno,
									ostream &os) const {

	multimap<string, LineOfSourceCode *>::size_type cnt = hdb.count(name);
	if (cnt <= 1) {	// 0 when not found, 1 when found self
		return;
	}

	bool printOnlyNumbers = false;
	multimap<string, LineOfSourceCode *>::const_iterator cit = hdb.find(name);

	// it's impossible because we already check this with count() above
	assert(cit != hdb.end());

	for ( ; cnt > 0; ++cit, --cnt) {
		if (lineno <= cit->second->getLineNumber()) {
			continue;
		}
		if (!printOnlyNumbers) {
			os << " (already included at line ";
			printOnlyNumbers = true;
		} else {
			os << ", ";
		}
		os << cit->second->getLineNumber();
	}

	if (printOnlyNumbers) {
		os << ')';
	}


}

void
SourceFile::printHeaderNames(bool printLineNumbers, size_t level, bool pheader) const {

	if (printHeader) {
		if (sourceFile.empty()) {
			cout << fileName << " not include headers." << endl;
			return;
		} else {
			cout << fileName << " include headers:" << endl;
		}
	}

	for (list<LineOfSourceCode *>::const_iterator lst = sourceFile.begin();
			lst != sourceFile.end();
			++lst) {

		if ((*lst)->isSystemInclude() || (*lst)->isUserInclude()) {

			cout << string(level*2, ' ');

			if (printLineNumbers) {
				cout << ':' << (*lst)->getLineNumber();
			}

			cout << ": " << (*lst)->getHeaderName();

			if ((*lst)->isUserInclude()) {
				if (!recursive_mode) {
					cout << " (user defined)";
				} else if(!(*lst)->usf->fileExist()) {
					cout << " (not found)";
				}
			}

			if (pheader) {
				printDuplicateHeaders((*lst)->getHeaderName(), (*lst)->getLineNumber(), cout);
			}

			cout << endl;

			if ((*lst)->isUserInclude() && recursive_mode && (*lst)->usf->fileExist()) {
				(*lst)->usf->printHeaderNames(printLineNumbers, level+1);
			}

		}
	}
}
