/*
 * Copyright (C) 2007-2008 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include "LineOfSourceCode.hh"	// for SourceFile and LineOfSourceCode classes

#include <cassert>	// for assert()
#include <cctype>	// for isalnum()
#include <cstring>	// for strncmp(), strchr(), strlen(), strspn()


using std::string;
using std::cout;
using std::endl;

LineOfSourceCode::LineOfSourceCode(string line, size_t LineNumber)
	: sourceLine(line), lineno(LineNumber), status(NOTCHANGED), usf(NULL) {

	const char *p;
	const char *pp = NULL;

#ifdef FEATURE
	// It's impossible because getline() always returns string wo/ newline symbol
	assert(line.find_first_of('\n') == string::npos);

	if (line.empty() || line.find_first_not_of(" \t") == string::npos){
		type = BLANK;

	} else if ((p = _isDirective(line.c_str())) == NULL) {
#else
	if ((p = _isDirective(line.c_str())) == NULL) {
#endif
		type = CODE;

	} else if ((pp = _detectDirective(p, "include")) != NULL) {

		if (_isHeader(pp, '"', '"')) {
			type = USRHDR;
		} else if (_isHeader(pp)) {
			type = SYSHDR;
		} else {
			type = UNKHDR;
		}

#ifdef FEATURE
	} else if (_detectDirective(p, "ifdef")) {
		type = IFDEF;

	} else if (_detectDirective(p, "ifndef")) {
		type = IFNDEF;

	} else if (_detectDirective(p, "endif")) {
		type = ENDIF;
#endif

	} else {
		type = UNKHDR;
	}

	if (isSystemInclude()) {
		const char *pos = strchr(++pp, '>');

		// it's impossible because isSystemInclude() already test this
		assert(pos != NULL);

		headerName = string(pp, pos-pp);

	} else if (isUserInclude()) {
		const char *pos = strchr(++pp, '"');

		// it's impossible because isUserInclude() already test this
		assert(pos != NULL);

		headerName = string(pp, pos-pp);
	}

}

LineOfSourceCode::~LineOfSourceCode() {

	if (isUserInclude() && usf != NULL) {
		delete usf;
	}
}

#define SKIP_SPACES(str) str += strspn(str, " \t")

const char *
LineOfSourceCode::_isDirective(const char *str) const {

	SKIP_SPACES(str);

	if (*str != '#') {
		return NULL;
	}

	str++;
	SKIP_SPACES(str);

	return str;
}

const char *
LineOfSourceCode::_detectDirective(const char *str, const char *directiveName) const {

	size_t size = strlen(directiveName);

	if (strncmp(str, directiveName, size) != 0) {
		return NULL;
	}

	str += size;
	SKIP_SPACES(str);

	return str;
}

#undef SKIP_SPACES

bool
LineOfSourceCode::_isHeader(const char *str, char begin, char end) const {

	if (*str != begin) {
		return false;
	}
	str++;

	// TODO: recognize more symbols in header name
	while (isalnum(*str) || *str == '.'
		|| *str == '/' || *str == '_'
		|| *str == '-' || *str == '+') {
		str++;
	}

	if (*str != end) {
		return false;
	}

	return true;
}
