/*
 * Copyright (C) 2007-2008 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <algorithm>// for sort()
#include <cassert>	// for assert()
#include <cstdlib>	// for exit() and EXIT_FAILURE constant
#include <fstream>	// for ifstream

#include "LineOfSourceCode.hh"	// for LineOfSourceCode and SourceFile classes
#include "Patch.hh"				// for Patch class

using std::ifstream;
using std::string;
using std::vector;
using std::multimap;
using std::cerr;
using std::cout;
using std::endl;

// Sun Studio compiler need it
using std::sort;


BaseSourceFile::BaseSourceFile(const std::string &filename) : fileName(filename) {

	// We can't use dirname() here because Windows doesn't have her
#ifndef _WIN32
	string::size_type pos = filename.find_last_of('/');
#else
	string::size_type pos = filename.find_last_of('\\');
#endif

	if (pos != string::npos){
		filePath = filename.substr(0, pos+1);
	}

}

BaseSourceFile::~BaseSourceFile() {

	for (vector<LineOfSourceCode *>::const_iterator vec = sourceFile.begin();
			vec != sourceFile.end();
			++vec) {
		delete *vec;
	}
}

int
BaseSourceFile::parseSourceFile(const string &filename) {

	ifstream file(filename.c_str());
	if (!file) {
		return 1;
	}

	string line;
	size_t lineno = 1;
	while (getline(file, line)) {
		try {
				LineOfSourceCode *losc = new LineOfSourceCode(line, lineno++);

				try {
					sourceFile.push_back(losc);
					
					if (losc->isInclude()) {
						headers.push_back(losc);
					}
				
				} catch (...) {
					delete losc;
					throw;
				}

		} catch (...) {
			file.close();
			return 1;
		}
	}

	file.close();
	return 0;
}


DirectSourceFile::DirectSourceFile(const string &filename, bool recursive)
				: BaseSourceFile(filename), rflag(recursive) {

	int ret = parseSourceFile(fileName);
	if (ret) {
		cerr << "Can't open file " << fileName << endl;
		exit(EXIT_FAILURE);
	}

	for (vector<LineOfSourceCode *>::const_iterator vec = sourceFile.begin();
			vec != sourceFile.end();
			++vec) {
		if ((*vec)->isInclude()) {
			addHeader(*vec);

			if (rflag && (*vec)->isUserInclude()) {
				IndirectSourceFile *usf = new IndirectSourceFile(filePath + (*vec)->getHeaderName());
				(*vec)->usf = usf;
			}
		}
	}

}

void
DirectSourceFile::printHeaderTree(bool nflag, size_t level) const {

	if (sourceFile.empty()) {
		cout << fileName << " not include headers." << endl;
		return;
	} else {
		cout << fileName << " include headers:" << endl;
	}

	for (vector<LineOfSourceCode *>::const_iterator vec = headers.begin();
			vec != headers.end();
			++vec) {

		cout << string(level*2, ' ');

		if (nflag) {
			cout << ':' << (*vec)->getLineNumber();
		}

		cout << ": " << (*vec)->getHeaderName();

		if ((*vec)->isUserInclude()) {
			if (!rflag) {
				cout << " (user defined)";
			} else if(!(*vec)->usf->fileExist()) {
				cout << " (not found)";
			}
		}

		printDuplicateHeaders((*vec)->getHeaderName(), (*vec)->getLineNumber());

		cout << endl;

		if (rflag && (*vec)->isUserInclude() && (*vec)->usf->fileExist()) {
			(*vec)->usf->printHeaderTree(nflag, level+1);
		}

	}
}

// TODO: split to 2 functions and move one of them to Patch class
void
DirectSourceFile::producePatch() {
	
	// vector with numbers of lines marked for deletion
	vector<size_t> deleted_lines_list;
	
	for (vector<LineOfSourceCode *>::const_iterator vec = headers.begin();
			vec != headers.end();
			++vec) {
		
		// only for not touching yet headers
		if ((*vec)->getStatus() != DELETED) {
			
			// search duplicates for current string
			// 0 when not found, 1 when found self
			multimap<string, LineOfSourceCode *>::size_type cnt = hdb.count((*vec)->getHeaderName());
			
			// we always search for really existing headers
			assert(cnt != 0);
			
			// duplicate existing
			if (cnt > 1) {
				
				multimap<string, LineOfSourceCode *>::const_iterator cit = hdb.find((*vec)->getHeaderName());
				
				// it's impossible because we already check this with count() above
				assert(cit != hdb.end());
				
				// this loop mark *all* duplicated headers after first
				for ( ; cnt > 0; ++cit, --cnt) {
					
					if (cit->second->getLineNumber() <= (*vec)->getLineNumber()) {
						// skip yourself and first unique header
						continue;
					}
					
					// assign status for deletion for founded duplicate
					cit->second->setStatus(DELETED);
					deleted_lines_list.push_back(cit->second->getLineNumber());
				}
			}
		}
	}
	
	// if duplicates wasn't found
	if (deleted_lines_list.size() == 0) {
		// nothing interesting found for producing patch
		return;
	}
	
	// print header for future patch
	Patch::printHeader(fileName);
	
	// TODO: define/constant?
	const size_t CONTEXT = 3;
	
	// how may lines should be prints after current line
	size_t context_after = 0;
	
	// for calculates result range we need know how many lines was
	// removed in all previous hunks.
	size_t lines_deleted_in_hunks = 0;
	
	// number of duplicated lines may be in wrong order. For example
	// when first duplicate located at the end of file and second
	// duplicate located at the middle.
	sort(deleted_lines_list.begin(), deleted_lines_list.end());
	
	vector<size_t>::const_iterator current_deleted_line = deleted_lines_list.begin();
	
	// again go over all lines in file
	for (vector<LineOfSourceCode *>::const_iterator vec = sourceFile.begin();
			vec != sourceFile.end();
			++vec) {
		
			// only if we found header which was marked as duplicate
			if ((*vec)->isInclude() && (*vec)->getStatus() == DELETED) {
				
				// print hunk header only for lines which begin hunk.
				// For other deleted lines hunk header was already
				// printed.
				if (current_deleted_line != deleted_lines_list.end() &&
					(*vec)->getLineNumber() == *current_deleted_line) {
					// calculate hunk coordinates
					// TODO: add link with describing of unified diff format
					
					// counter for lines which will deleted in current
					// hunk
					size_t lines_deleted_in_current_hunk = 1;
					
					// counter for lines between deleted headers
					size_t not_affected_lines = 0;
					
					// count lines which deleted in current hunk and
					// set current_deleted_line to first deleted
					// header in next hunk
					// NOTE: we start counter from next deleted header
					for (vector<size_t>::const_iterator hvec = current_deleted_line + 1;
						hvec != deleted_lines_list.end() &&
						(*hvec - *current_deleted_line - 1) <= CONTEXT + CONTEXT;
						++hvec) {
						
						// if diff equal to zero then hvec == current_deleted_line
						// if diff less then zero then
						// deleted_lines_list array wasn't sorted
						assert(*hvec - *current_deleted_line > 0);
						
						// calculate our bottom context and top
						// context for next deleted header in current
						// hunk
						size_t lines_between_deleted_headers = *hvec - *current_deleted_line - 1;
						
						if (lines_between_deleted_headers > CONTEXT) {
							context_after = lines_between_deleted_headers;
						}
						
						if (lines_between_deleted_headers > 0) {
							not_affected_lines += lines_between_deleted_headers;
						}
						
						++lines_deleted_in_current_hunk;
						++current_deleted_line;
						
					}
					
					// default values which uses for first 3 lines:
					// ranges starts from first line
					size_t from_range_start = 1;
					size_t to_range_start = 1;
					if ((*vec)->getLineNumber() > CONTEXT) {
						from_range_start = (*vec)->getLineNumber() - CONTEXT;
						to_range_start = (*vec)->getLineNumber() - CONTEXT;
					}
					
					// only for second and more hunks: take into
					// account about already removed lines
					if (lines_deleted_in_hunks > 0) {
						to_range_start -= lines_deleted_in_hunks;
					}
					
					lines_deleted_in_hunks += lines_deleted_in_current_hunk;
					
					// calculate how many lines will prints before
					size_t top_context;
					if ((*vec)->getLineNumber() > CONTEXT) {
						top_context =  CONTEXT;
					} else {
						// don't count current lines
						top_context = (*vec)->getLineNumber() - 1;
					}
					
					// calculate how many lines will prints after
					size_t bottom_context;
					// we calculate bottom context start from last
					// (not current!) deleted header.
					if (*current_deleted_line + CONTEXT < sourceFile.size()) {
						bottom_context = CONTEXT;
					} else {
						bottom_context = sourceFile.size() - *current_deleted_line;
					}
					
					size_t to_range_count = not_affected_lines + top_context + bottom_context;
					size_t from_range_count = lines_deleted_in_current_hunk + to_range_count;
					
					// print hunk header
					cout << "@@ -"
						<< from_range_start
						<< ','
						<< from_range_count
						<< " +"
						<< to_range_start
						<< ','
						<< to_range_count
						<< " @@"
						<< endl;
					
					current_deleted_line++;
					
					// print context (3 or less lines before current line)
					// if needed
					vector<LineOfSourceCode *>::const_iterator rvec;
					if ((*vec)->getLineNumber() > CONTEXT) {
						rvec = vec - CONTEXT;
					} else {
						// special case for first 3 lines
						rvec = sourceFile.begin();
					}
					
					for ( ; rvec != vec; ++rvec) {
						// TODO: use overloading operator
						cout << ' ' << (*rvec)->toString() << endl;
					}
				}
				
				// will print full context after printing current
				// line
				// if context_after > CONTEXT then leave this value as
				// is (needed for printing top context for next
				// deleted headers in current hunk)
				if (context_after < CONTEXT) {
					context_after = CONTEXT;
				}
				
				// print also current line which should be deleted
				cout << '-' << (*vec)->toString() << endl;
				
			} else { // for all other lines and not duplicated headers
			
				if (context_after > 0) {
					cout << ' ' << (*vec)->toString() << endl;
					context_after--;
				}
			}
		
	}
	
}

void
DirectSourceFile::addHeader(LineOfSourceCode *losc) {
	// NOTE:
	// we use multimap::value_type instead of make_pair() because it
	// allow building program under Sun Studio compiler
	hdb.insert(multimap<string, LineOfSourceCode *>::value_type(losc->getHeaderName(), losc));
}

void
DirectSourceFile::printDuplicateHeaders(const string &name, size_t lineno) const {

	multimap<string, LineOfSourceCode *>::size_type cnt = hdb.count(name);
	
	// we always search for really existing headers
	assert(cnt != 0);
	
	if (cnt <= 1) {	// 0 when not found, 1 when found self
		return;
	}

	bool printOnlyNumbers = false;
	multimap<string, LineOfSourceCode *>::const_iterator cit = hdb.find(name);

	// it's impossible because we already check this with count() above
	assert(cit != hdb.end());

	for ( ; cnt > 0; ++cit, --cnt) {
		if (lineno <= cit->second->getLineNumber()) {
			// skip yourself and first unique header
			continue;
		}
		if (!printOnlyNumbers) {
			cout << " (already included at line ";
			printOnlyNumbers = true;
		} else {
			cout << ", ";
		}
		cout << cit->second->getLineNumber();
	}

	if (printOnlyNumbers) {
		cout << ')';
	}

}


IndirectSourceFile::IndirectSourceFile(const string &filename)
				: BaseSourceFile(filename), fileWasOpened(false) {

	int ret = parseSourceFile(fileName);
	if (ret) {
		return;
	}

	fileWasOpened = true;

	for (vector<LineOfSourceCode *>::const_iterator vec = sourceFile.begin();
		vec != sourceFile.end();
		++vec) {
		if ((*vec)->isUserInclude()) {
			IndirectSourceFile *usf = new IndirectSourceFile(filePath + (*vec)->getHeaderName());
			(*vec)->usf = usf;
		}
	}
}

void
IndirectSourceFile::printHeaderTree(bool nflag, size_t level) const {

	for (vector<LineOfSourceCode *>::const_iterator vec = headers.begin();
			vec != headers.end();
			++vec) {

		cout << string(level*2, ' ');

		if (nflag) {
			cout << ':' << (*vec)->getLineNumber();
		}

		cout << ": " << (*vec)->getHeaderName();

		if ((*vec)->isUserInclude() && !(*vec)->usf->fileExist()) {
			cout << " (not found)";
		}

		cout << endl;

		if ((*vec)->isUserInclude() && (*vec)->usf->fileExist()) {
			(*vec)->usf->printHeaderTree(nflag, level+1);
		}

	}
}
