/*
 * Copyright (C) 2008 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <iostream>	// for cerr, cout and endl
#include <cassert>	// for assert()
#include <cstdio>	// for perror()
#include <cstdlib>	// for exit() and EXIT_FAILURE constant
#include <ctime>	// for time(), localtime() and strftime()

// for stat()
#include <sys/types.h>
#include <sys/stat.h>
#ifndef _WIN32
// this header not needed for stat() on Windows
#include <unistd.h>
#endif

#include "Patch.hh"


using std::string;
using std::cerr;
using std::cout;
using std::endl;


string
Patch::convertTimeToDate(const time_t *time) {
	
	assert(time != NULL);
	
	struct tm *tmp = localtime(time);
	if (tmp == NULL) {
		perror("localtime");
		exit(EXIT_FAILURE);
	}
	
	// buffer for string + terminating null byte
	const size_t SIZE = sizeof("YYYY-MM-DD HH:MM:SS +0000");
	char *buff = new char[SIZE]();
	
#ifndef _WIN32
	// "%F %T %z" is equivalent to "%Y-%m-%d %H:%M:%S %z"
	size_t ret = strftime(buff, SIZE, "%F %T %z", tmp);
#else
	// Don't use %F and %T because VC++2005 don't understand it.
	// Also don't use %z because in Windows it returns time zone name.
	size_t ret = strftime(buff, SIZE, "%Y-%m-%d %H:%M:%S +0000", tmp);
#endif
	
	if (ret == 0) {
		cerr << "strftime: can't get date and time!" << endl;
		delete [] buff;
		exit(EXIT_FAILURE);
	}
	
	// From man strftime(3):
	// The strftime() function returns the number of characters placed
	// in  the array s, not including the terminating null byte.
	assert(ret == SIZE-1);
	
	string date(buff);
	
	delete [] buff;
	
	return date;
}

string
Patch::getFileModificationDate(const string &fileName) {
	
	int ret;
	struct stat fileStat;
	
	// get file information (we need only st_mtime member)
	ret = stat(fileName.c_str(), &fileStat);
	if (ret == -1) {
		perror("stat");
		exit(EXIT_FAILURE);
	}
	
	return convertTimeToDate(&fileStat.st_mtime);
}

string
Patch::getCurrentDate() {
	
	time_t t = time(NULL);
	if (t == (time_t)-1) {
		perror("time");
		exit(EXIT_FAILURE);
	}
	
	return convertTimeToDate(&t);
}

void
Patch::printHeader(const string &fileName) {
	
	string modificaction_date = getFileModificationDate(fileName);
	string current_date = getCurrentDate();
	
	// show our blurb
	printDescription();
	
	cout << "--- " << fileName << ".orig" << '\t' << modificaction_date << endl;
	cout << "+++ " << fileName << '\t' << current_date << endl;
}

void
Patch::printDescription() {
#ifndef PACKAGE_STRING
// this situation shouldn't happens because autotools define it for us
#error "PACKAGE_STRING wasn't defined!"
#else
	cout << "\n    Patch was generated by " << PACKAGE_STRING << "\n" << endl;
#endif
}

