#!/bin/sh

PASSES=0
FAILS=0

# hlint version (used in replace_version() function)
VERSION="0.3"

test_failed() {
	FAILS=`expr $FAILS + 1`
	printf '\033[31mfailed\033[0m\n'
}

test_passed() {
	PASSES=`expr $PASSES + 1`
	printf '\033[32mok\033[0m\n'
}

compare_files() {
	if [ $# -ne 2 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	ORIG=$1
	MODIF=$2
	
	cmp -s $ORIG $MODIF
	if  [ $? -ne 0 ]; then
		test_failed
		diff -u $ORIG $MODIF
		return 1
	fi
	
	return 0
}

# Try to apply patch which was generated by hlint. Get number of
# generated .out file.
try_to_apply_patch() {
	if [ $# -ne 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	TESTNO="$1"
	
	patch --silent --dry-run <"$TESTNO.out"
	if [ $? -ne 0 ]; then
		test_failed
		echo "Generated patch not applicable to file!" >&2
		return 1
	fi
	
	return 0
}

do_test() {
	if [ $# -lt 2 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	NO="$1"
	MSG="$2"
	TESTNO=`printf '%03d\n' $NO`
	
	if [ $# -eq 3 -a "$3" = "patch" ]; then
		replace_date $TESTNO
		replace_version $TESTNO
	fi
	
	printf '%3d Test %s... ' "$NO" "$MSG"
	
	compare_files ${TESTNO}.ok ${TESTNO}.out
	if [ $? -ne 0 ]; then
		rm -f ${TESTNO}.out
		return
	fi
	
	# SunOS version of patch cannot apply patch without change file
	# (like GNU patch with --dry-run option). So just disable this
	# test.
	# XXX: user may have installation GNU version of patch
	if [ $# -eq 3 -a "$3" = "patch" -a "`uname`" != "SunOS" ]; then
		try_to_apply_patch $TESTNO
		if [ $? -ne 0 ]; then
			rm -f ${TESTNO}.out
			return
		fi
	fi
	
	test_passed
	
	rm -f ${TESTNO}.out
}

run_hlint() {
	if [ $# -lt 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	TESTNO="$1"; shift
	
	$HLINT "$@" "$TESTNO.in" >"`basename $TESTNO`.out" 2>&1
}

# Replace current date in file specified by number to many X symbols.
replace_date() {
	if [ $# -ne 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	TESTNO="$1"
	
	perl -i -pe 's|\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2} ([-+])\d{4}|XXXX-XX-XX XX:XX:XX \1XXXX|' \
		$TESTNO.out
}

# Replace current version of hlint in file specified by number to X
# symbols.
replace_version() {
	if [ $# -ne 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	TESTNO="$1"
	
	perl -i -pe "s|hlint $VERSION|hlint X.X|" $TESTNO.out
}

HLINT=../src/hlint

if [ ! -x "$HLINT" ]; then
	echo "File $HLINT not found! Please run 'make' before start tests!" >&2
	exit 1
fi


$HLINT >001.out 2>&1
do_test 1 "without arguments"

run_hlint 002 002.ok
do_test 2 "with two files in command line"

run_hlint 003
do_test 3 "when main file not exist"

run_hlint 004
do_test 4 "with empty source file"

run_hlint 005
do_test 5 "system headers detection"

run_hlint 006 -n
do_test 6 "system header detection with -n flag"

run_hlint 007
do_test 7 "user header detection"

run_hlint 008 -r
do_test 8 "user header with -r flag"

run_hlint 009 -r
do_test 9 "not exist user header with -r flag"

run_hlint 010/010 -r
do_test 10 "when source file in another directory and -r flag specified"

run_hlint 011 -r
do_test 11 "when we have two similar users headers and can't open second"

$HLINT -r -n >012.out 2>&1
do_test 12 "when user specified only options without file name"

$HLINT -X >013.out 2>&1
do_test 13 "when user specified unknown option"

run_hlint 014 -rn
do_test 14 "for works combined options"

run_hlint 015 -r
do_test 15 "detection of duplicate headers"

run_hlint 016 -r
do_test 16 "deep recursion"

run_hlint 017 -p
do_test 17 "-p option: simple example (duplicate at middle of code)" patch

run_hlint 018 -p
do_test 18 "-p option: duplicate at second line" patch

run_hlint 019 -p
do_test 19 "-p option: duplicate at before last line" patch

run_hlint 020 -p
do_test 20 "-p option: two duplicates at neighbourhood lines" patch

run_hlint 021 -p
do_test 21 "-p option: duplicate after two lines" patch

run_hlint 022 -p
do_test 22 "-p option: produce empty file if duplicates not found"

run_hlint 023 -rpn
do_test 23 "-p option: conflicts with -r and -n options"

run_hlint 024 -p
do_test 24 "-p option: sequence of headers across one line" patch

run_hlint 025 -p
do_test 25 "-p option: situation when first duplicates early than second" patch

run_hlint 026 -p
do_test 26 "-p option: decrease bottom context if regular strings between deleted headers" patch

run_hlint 027 -p
do_test 27 "-p option: don't create two hunks when betwen deleted headers 6 lines" patch


echo
printf '    Tests passes:\t%2d\n' $PASSES
printf '    Tests failures:\t%2d\n' $FAILS
echo

