/*
 * Copyright (C) 2007-2009 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DirectSourceFile.hh"
#include "DirectiveWithCondition.hh"
#include "Patch.hh"				// for Patch class
#include "UserHeader.hh"

#include <algorithm>// for sort() and equal()
#include <cassert>	// for assert()
#include <iostream>	// for cerr, cout, endl
#include <sstream>	// for ostringstream
#include <stdexcept>// for runtime_error exception
#include <utility>	// for pair type

#ifdef ENABLE_BOOST
#include <boost/foreach.hpp>
#include <boost/lambda/lambda.hpp>
using boost::lambda::_1;
using boost::lambda::_2;
#endif

using std::cerr;
using std::cout;
using std::endl;
using std::equal;
using std::map;
using std::ostringstream;
using std::pair;
using std::runtime_error;
using std::set;
using std::sort;
using std::string;
using std::vector;

namespace {

#ifndef ENABLE_BOOST
// Used as < operator for sorting pointers to DirectiveWithCondition
bool lessForPointersToDirectivesWithCondition(const DirectiveWithCondition *lhs, const DirectiveWithCondition *rhs) {
	return *lhs < *rhs;
}

// Used as == operator for compare pointers to DirectiveWithCondition
bool equalForPointersToDirectivesWithCondition(const DirectiveWithCondition *lhs, const DirectiveWithCondition *rhs) {
	return *lhs == *rhs;
}
#endif

} // end of anonymous namespace


/**
 * @param[in] filename file which will parsed
 * @param[in] recursive process or not user's headers
 * @exception syntax_error when \#include nested too deeply
 * @exception std::runtime_error when source file contains syntax errors
 **/
DirectSourceFile::DirectSourceFile(const std::string &filename, bool recursive)
				: SourceFile(filename), rflag(recursive) {

	parseSourceFile();

#ifndef ENABLE_BOOST
	for (vector<LineOfSourceCode *>::const_iterator vec = sourceFile.begin();
			vec != sourceFile.end();
			++vec) {
		
		LineOfSourceCode *losc = *vec;
#else
	foreach (LineOfSourceCode *losc, sourceFile) {
#endif
		
		if (rflag && losc->isUserInclude()) {
			try {
				UserHeader *usrhdr = static_cast<UserHeader *>(losc);
				IndirectSourceFile *usf = new IndirectSourceFile(
							filePath + usrhdr->getHeaderName(),
							SourceFile::RECURSIVE_MODE_DEFAULT_DEEP);
				usrhdr->setSourceFile(usf);
			} catch (const runtime_error &e) {
				throw syntax_error(e.what(), losc->getLineNumber());
			}
		}
		
	}
	
	vector<syntax_error> parsing_errors;
	condition_pairs = Algorithm::getDirectivesPairs(this, parsing_errors);
	
	if (!parsing_errors.empty()) {
#ifndef ENABLE_BOOST
		for (vector<syntax_error>::const_iterator vec = parsing_errors.begin();
				vec != parsing_errors.end();
				++vec) {
		
			const syntax_error &err = *vec;
#else
		foreach (const syntax_error &err, parsing_errors) {
#endif
			cerr << getFileName() << ": syntax error at line " << err.where() << ": " << err.what() << endl;
		}
		
		throw runtime_error("");
	}
}

void
DirectSourceFile::printHeaderTree(bool nflag, size_t level) const {
	
	vector<CHdrPtr> all_headers =
		Algorithm::getHeaders(this);
	
	if (all_headers.empty()) {
		cout << getFileName() << " not include headers." << endl;
		return;
	} else {
		cout << getFileName() << " include headers:" << endl;
	}
	
	vector<Algorithm::DuplPair> duplicated_headers =
		Algorithm::getDuplicatedHeadersPairs(all_headers);
	
	// in some cases user's headers may has names similar to
	// headers from system and they are not duplicates.
	// See also test #089
	duplicated_headers =
		Algorithm::excludeHeadersWithDifferentTypes(duplicated_headers);
	
#ifndef ENABLE_BOOST
	for (vector<CHdrPtr>::const_iterator vec = all_headers.begin();
			vec != all_headers.end();
			++vec) {
		
		CHdrPtr hdr = *vec;
#else
	foreach (CHdrPtr hdr, all_headers) {
#endif
		
		ostringstream line;
		line << string(level*2, ' ');

		if (nflag) {
			line << ":" << hdr->getLineNumber();
		}

		line << ": " << hdr->getHeaderName();

		if (hdr->isUserInclude()) {
			const UserHeader *usrhdr =
				static_cast<const UserHeader *>(hdr);
			if (!rflag) {
				line << " (user defined)";
			} else if(!(usrhdr->getSourceFile())->isFileExists()) {
				line << " (not found)";
			}
		}
		
		bool printOnlyNumbers = false;
		vector<size_t> dup_lines_before =
			Algorithm::getEarlyIncludedHeaders(hdr, duplicated_headers);
		
		for (vector<size_t>::const_iterator cit = dup_lines_before.begin();
				cit != dup_lines_before.end();
				++cit) {
			
			if (!printOnlyNumbers) {
				line << " (already included at line ";
				printOnlyNumbers = true;
			} else {
				line << ", ";
			}
			
			line << *cit;
		}

		if (printOnlyNumbers) {
			line << ')';
		}
		
		cout << line.str() << endl;

		if (rflag && hdr->isUserInclude()) {
			const UserHeader *usrhdr =
				static_cast<const UserHeader *>(hdr);
			if (usrhdr->getSourceFile()->isFileExists()) {
				usrhdr->getSourceFile()->printHeaderTree(nflag, level+1);
			}
		}

	}
}

void
DirectSourceFile::markDuplicates(std::set<size_t> &deleted_lines_list) const {
	
	vector<CHdrPtr> all_headers =
		Algorithm::getHeaders(this);
	
	vector<Algorithm::DuplPair> duplicated_headers =
		Algorithm::getDuplicatedHeadersPairs(all_headers);
	
	// in some cases user's headers may has names similar to
	// headers from system and they are not duplicates.
	// See also test #043
	duplicated_headers =
		Algorithm::excludeHeadersWithDifferentTypes(duplicated_headers);
	
#ifndef ENABLE_BOOST
	for (vector<Algorithm::DuplPair>::const_iterator vec = duplicated_headers.begin();
			vec != duplicated_headers.end();
			++vec) {
		
		Algorithm::DuplPair pair = *vec;
#else
	foreach (Algorithm::DuplPair pair, duplicated_headers) {
#endif
		
		CHdrPtr original  = pair.first;
		CHdrPtr duplicate = pair.second;
		
		const size_t original_line_number  = original->getLineNumber();
		const size_t duplicate_line_number = duplicate->getLineNumber();
		
		// Shouldn't happens because we always have sorted lines
		assert(original_line_number < duplicate_line_number);
		
		vector<Algorithm::CDrtvPtr> orig =
			Algorithm::getConditionsAffectsToLine(condition_pairs, original_line_number);
		
		vector<Algorithm::CDrtvPtr> dupl =
			Algorithm::getConditionsAffectsToLine(condition_pairs, duplicate_line_number);
		
		// Order of nested conditions shouldn't affect result.
		// See also test #082
		if (orig.size() > 1 && orig.size() == dupl.size()) {
#ifndef ENABLE_BOOST
			sort(orig.begin(), orig.end(), lessForPointersToDirectivesWithCondition);
			sort(dupl.begin(), dupl.end(), lessForPointersToDirectivesWithCondition);
#else
			sort(orig.begin(), orig.end(), *_1 < *_2);
			sort(orig.begin(), orig.end(), *_1 < *_2);
#endif
		}
		
		// if original and duplicated line located inside
		// different conditions then don't mark second as
		// duplicate of first.
		//
		// Emulate != operator for vector with pointers to DirectiveWithCondition
		if (!(orig.size() == dupl.size() &&
#ifndef ENABLE_BOOST
			equal(orig.begin(), orig.end(), dupl.begin(), equalForPointersToDirectivesWithCondition)
#else
			equal(orig.begin(), orig.end(), dupl.begin(), *_1 == *_2)
#endif
			)) {
			
			// handle situation like shown in tests #046 and #047:
			// if first duplicate located inside #ifdef and second
			// included absolutely then remove second duplicate
			// and #if's directives around first.
			if (!orig.empty()) {
				
				if (dupl.empty()) {
					// try to determine: is first duplicate alone
					// in #if directive or not?
					set<size_t> deleted_test_list;
					deleted_test_list.insert(original_line_number);
					markLinesAroundDuplicatesRecursively(deleted_test_list);
					// first duplicate alone in #if directive
					if (deleted_test_list.size() > 1) {
						// add to list of removed lines only
						// directives without self first duplicate
						deleted_test_list.erase(original_line_number);
						
						// don't remove single #else directive if
						// first duplicate located inside
						// #else/#endif pair.
						// See also test #052
						if (deleted_test_list.size() == 1) {
							// we expect this single element is
							// #else directive
							continue;
						}
						
						deleted_lines_list.insert(
								deleted_test_list.begin(),
								deleted_test_list.end());
					} else {
						// don't delete anything.
						// See also test #047
						continue;
					}
				} else {
					// if first header included not absolutely
					// See also test #042
					if (dupl.size() > orig.size()) {
						vector<Algorithm::CDrtvPtr> last_elements(
								dupl.begin() + (dupl.size() - orig.size()),
								dupl.end());
						// Emulate != operator for vector with pointers to DirectiveWithCondition
						if (!(orig.size() == last_elements.size()
#ifndef ENABLE_BOOST
							&& equal(orig.begin(), orig.end(), last_elements.begin(), equalForPointersToDirectivesWithCondition)
#else
							&& equal(orig.begin(), orig.end(), last_elements.begin(), *_1 == *_2)
#endif
							)) {
							continue;
						}
						// if orig == last_elements then second
						// should be marked as duplicate
					} else {
						continue;
					}
				}
			}
		}
		
		// for test #044: when original header and it duplicate
		// inside one #ifdef/#endif, but separated by #else
		// directive. Don't treat it as duplicates.
		if (orig.size() == dupl.size()) {
			
			// Shouldn't happens because we already test it above
			// equal() emulate == operator for vector with pointers to DirectiveWithCondition
#ifndef ENABLE_BOOST
			assert(equal(orig.begin(), orig.end(), dupl.begin(), equalForPointersToDirectivesWithCondition));
#else
			assert(equal(orig.begin(), orig.end(), dupl.begin(), *_1 == *_2));
#endif
			
			size_t i;
			size_t nested_ifs_counter = 0;
			for (i = getNumberOfNextLine(original_line_number);
					i < duplicate_line_number;
					++i) {
				
				LineType type = getLineOfSourceCodeClass(i)->getType();
				
				if (type == UNDEF) {
					// between original and duplicated header
					// exists #undef, so it's possible intended
					// duplication (see test #071)
					break;
				
				} else if (type == IF || type == IFDEF || type == IFNDEF) {
					++nested_ifs_counter;
				
				} else if (type == ELSE || type == ELIF) {
					// between original and duplicated header
					// exists #else, so they are not duplicated
					// and this #else belong to our opened #if
					if (nested_ifs_counter == 0) {
						break;
					}
				}
			}
			
			if (i != duplicate_line_number) {
				continue;
			}
		}
		
		// assign status for deletion for founded duplicate
		deleted_lines_list.insert(duplicate_line_number);
	}
}

bool
DirectSourceFile::markLinesAroundDuplicates(std::set<size_t> &deleted_lines_list) const {
	
	// don't continue if duplicates not found
	if (deleted_lines_list.empty()) {
		return false;
	}
	
	// we can't add new deleted lines to set during loop over him.
	// So we use temporary vector for saving new just deleted lines
	set<size_t> addition_deleted_lines;
	
	for (set<size_t>::const_iterator current_line = deleted_lines_list.begin();
			current_line != deleted_lines_list.end();
			++current_line) {
		
		// don't process another directives or blank lines which may
		// added to list by markDuplicates() function.
		if (! getLineOfSourceCodeClass(*current_line)->isInclude()) {
			continue;
		}
		
		/*
		 * Search before deleted header
		 **/
		
		// how many lines will be removed
		// if 0 then don't delete anything
		size_t remove_before = 0;
		
		// when we remove duplicate inside #else/#elif and #endif we
		// shouldn't remove #endif because it needed for first
		// #if/#ifdef/#ifndef
		bool dont_delete_endif = false;
		
		for (size_t i = getNumberOfPreviousLine(*current_line); i >= getNumberOfFirstLine(); --i) {
			
			if (isDeletedLine(deleted_lines_list, i)) {
				// don't process lines already marked for deletion
				++remove_before;
				continue;
			}
			
			switch (getLineOfSourceCodeClass(i)->getType()) {
				
				case BLANK:
				case COMMENT:
					// delete empty lines and comments too
					++remove_before;
					continue;
				
				case IF:
				case IFDEF:
				case IFNDEF:
					// we reach first #if/#ifdef/#ifndef and can stop
					++remove_before;
					break;
				
				case ELSE:
				case ELIF:
					// we reach first #else/#elif and can stop
					++remove_before;
					dont_delete_endif = true;
					break;
				
				default:
					// we found regular line or some directive and can
					// stop
					remove_before = 0;
					break;
			}
			
			// we are there only when switch was break'ed.
			// quit from for() loop
			break;
		}
		
		if (remove_before == 0) {
			// not needed removing lines before, so don't search
			// after. Go to next duplicate
			continue;
		}
		
		/*
		 * Search after deleted header
		 **/
		
		// how many lines will be removed
		// if 0 then don't delete anything
		size_t remove_after = 0;
		
		for (size_t i = getNumberOfNextLine(*current_line); i <= getNumberOfLastLine(); ++i) {
			
			if (isDeletedLine(deleted_lines_list, i)) {
				// don't process lines already marked for deletion
				++remove_after;
				continue;
			}
			
			switch (getLineOfSourceCodeClass(i)->getType()) {
				
				case BLANK:
				case COMMENT:
					// delete empty lines and comments too
					++remove_after;
					continue;
				
				case ENDIF:
					// we found #endif and can stop
					// NOTE: not increase remove_after because this #endif
					// needed for top level #if/#ifdef/#ifndef, we remove
					// only #else/#elif
					if (!dont_delete_endif) {
						++remove_after;
					}
					break;
				
				default:
					// we found regular line or some directive and can
					// stop
					remove_after = 0;
					// don't remove lines before
					dont_delete_endif = false;
					break;
			}
			
			// we are there only when switch was break'ed.
			// quit from for() loop
			break;
		}
		
		if (remove_after == 0 && !dont_delete_endif) {
			// not needed removing lines after, so go to next line
			continue;
		}
		
		// add all lines before and after current duplicate (including
		// it) to set of removed lines
		for (size_t i = *current_line - remove_before; i <= *current_line + remove_after; ++i) {
			addition_deleted_lines.insert(i);
		}
	}

	if (addition_deleted_lines.empty()) {
		return false;
	}
	
	// add founded lines to end of main list of deleted lines
	deleted_lines_list.insert(addition_deleted_lines.begin(), addition_deleted_lines.end());
	
	return true;
}

void
DirectSourceFile::markLinesAroundDuplicatesRecursively(std::set<size_t> &deleted_lines_list) const {
	
	while (markLinesAroundDuplicates(deleted_lines_list)) {
		; // all work do markLinesAroundDuplicates() function
	}
}

void
DirectSourceFile::markEmptyLinesBetweenAlreadyRemovedLines(std::set<size_t> &deleted_lines_list) const {
	
	// we need as minimum two deleted lines
	if (deleted_lines_list.size() < 2) {
		return;
	}
	
	// we can't add new deleted lines to list during loop over him.
	// So we use temporary storage for saving new just deleted lines
	set<size_t> addition_deleted_lines;
	
	set<size_t>::const_iterator previous = deleted_lines_list.begin();
	
	// start from second element
	set<size_t>::const_iterator current = deleted_lines_list.begin();
	++current;
	
	for ( ; current != deleted_lines_list.end(); ++previous, ++current) {
		
		// lines numbers always sorted and hasn't duplicates
		assert(*previous < *current);
		
		// if current lines isn't next line
		if (*current == getNumberOfNextLine(*previous)) {
			continue;
		}
		
		bool all_lines_between_are_blank = true;
		
		// test if all lines between deleted are empty
		for (size_t i = getNumberOfNextLine(*previous); i < *current; ++i) {
			
			// exit if found first not empty line
			if (getLineOfSourceCodeClass(i)->getType() != BLANK) {
				all_lines_between_are_blank = false;
				break;
			}
			
		}
		
		// loop before wasn't terminating by break => all lines
		// are empty and may be removed
		if (all_lines_between_are_blank) {
			// schedule for removal
			for (size_t i = getNumberOfNextLine(*previous); i < *current; ++i) {
				addition_deleted_lines.insert(i);
			}
		}
	}
	
	// add founded lines to list of deleted lines
	deleted_lines_list.insert(addition_deleted_lines.begin(), addition_deleted_lines.end());
}

void
DirectSourceFile::invertIfdefsWithEmptyBody(
		std::set<size_t> &deleted_lines_list,
		std::set<size_t> &added_lines_list,
		std::map<size_t, size_t> &replaced_lines) const {
	// Because only this function adds elements to this set
	assert(added_lines_list.empty());
	
	// we need as minimum one duplicate
	if (deleted_lines_list.empty()) {
		return;
	}
	
	// we can't add new deleted lines to list during loop over him.
	// So we use temporary storage for saving new just deleted lines
	set<size_t> addition_deleted_lines;
	
	for (set<size_t>::const_iterator current_line = deleted_lines_list.begin();
			current_line != deleted_lines_list.end();
			++current_line) {
		
		// number of line where #if[n]def located.
		// If 0 then we haven't opened #if[n]def or we have another
		// not deleted line between #if[n]def and current line.
		size_t ifdef_before = 0;
		
		// Search before.
		//
		for (size_t i = getNumberOfPreviousLine(*current_line); i >= getNumberOfFirstLine(); --i) {
			
			// don't process deleted lines
			if (isDeletedLine(deleted_lines_list, i)) {
				continue;
			}
			
			switch (getLineOfSourceCodeClass(i)->getType()) {
				
				case IFDEF:
				case IFNDEF:
					// found first not removed #ifdef/#ifndef
					ifdef_before = i;
					break;
				
				default:
					break;
			}
			
			// we are there only when switch was break'ed.
			// quit from for() loop
			break;
		}
		
		// if we not alone inside #if[n]def or we not found opened
		// #if[n]def then go to process next line.
		if (ifdef_before == 0) {
			continue;
		}
		
		// number of line where #else located.
		// If 0 then #else not found or we not alone inside
		// #if[n]def/#else pair.
		size_t else_after = 0;
		
		// Search after.
		//
		for (size_t i = getNumberOfNextLine(*current_line); i <= getNumberOfLastLine(); ++i) {
			
			// don't process deleted lines
			if (isDeletedLine(deleted_lines_list, i)) {
				continue;
			}
			
			switch (getLineOfSourceCodeClass(i)->getType()) {
				
				case ELSE:
					// found first #else
					else_after = i;
					break;
				
				default:
					break;
			}
			
			// we are there only when switch was break'ed.
			// quit from for() loop
			break;
		}
		
		// if #else not found or we not alone inside #if[n]def/#else
		// pair then go to process next line.
		if (else_after == 0) {
			continue;
		}
		
		// added line with opened #if[n]def and line with #else
		// directive to list of removed lines
		addition_deleted_lines.insert(ifdef_before);
		addition_deleted_lines.insert(else_after);
		added_lines_list.insert(else_after);
		
		// Replace #else to !#if[n]def.)
		// NOTE:
		// we use map::value_type instead of make_pair() because it
		// allow building program under Sun Studio compiler
		pair<map<size_t, size_t>::iterator, bool> ret =
			replaced_lines.insert(map<size_t, size_t>::value_type(else_after, ifdef_before));
	
		// if map.insert() return false it means in our map already exists
		// elements which we want add, but it shouldn't happens because we
		// always adds only unique lines
		assert(ret.second == true);
	}
	
	// add founded lines to end of main list of deleted lines
	deleted_lines_list.insert(addition_deleted_lines.begin(), addition_deleted_lines.end());
}

void
DirectSourceFile::producePatch(bool print_banner) const {
	
	// set of numbers of lines marked for deletion
	set<size_t> deleted_lines_list;
	
	markDuplicates(deleted_lines_list);
	markLinesAroundDuplicatesRecursively(deleted_lines_list);
	markEmptyLinesBetweenAlreadyRemovedLines(deleted_lines_list);
	
	// set of numbers of lines marked for addition
	set<size_t> added_lines_list;
	
	map<size_t, size_t> replaced_lines_map;
	
	invertIfdefsWithEmptyBody(deleted_lines_list, added_lines_list, replaced_lines_map);
	
	Patch::produce(*this, deleted_lines_list, added_lines_list, replaced_lines_map, print_banner);
	
}

