/*
 * Copyright (C) 2008-2009 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Hunk.hh"
#include "DirectiveWithCondition.hh"
#include "DirectSourceFile.hh"

#include <algorithm>	// for min(), max(), min_element(), max_element()
#include <cassert>		// for assert()
#include <iostream>		// for cout, endl

using std::cout;
using std::endl;
using std::vector;
using std::map;
using std::min;
using std::min_element;
using std::max;
using std::max_element;

Hunk::Hunk(size_t lineno, bool add_for_deletion) {
	if (add_for_deletion) {
		pushLineForDeletion(lineno);
	} else {
		pushLineForAddition(lineno);
	}
}


void
Hunk::pushLineForDeletion(size_t lineno) {
	deleted_lines.push_back(lineno);
}

void
Hunk::pushLineForAddition(size_t lineno) {
	added_lines.push_back(lineno);
}

void
Hunk::printHunk(const DirectSourceFile &sf,
		const std::map<size_t, size_t> &replaced_lines,
		int lines_offset) const {
	
	const size_t from_range_start = getNumberOfFirstLine();
	const size_t to_range_start = getNumberOfFirstLine() + lines_offset;
	
	const size_t first_modified_line = getNumberOfFirstModifiedLine();
	const size_t last_modified_line  = getNumberOfLastModifiedLine();
	
	const size_t lines_deleted_in_hunk = deleted_lines.size();
	const size_t lines_added_in_hunk = added_lines.size();
	
	// by default we have only one modified line in hunk
	size_t not_affected_lines = 0;
	
	if (lines_deleted_in_hunk + lines_added_in_hunk >= 2) {
		// for case when we modify few lines in hunk
		not_affected_lines = last_modified_line - first_modified_line - lines_deleted_in_hunk + 1;
	}
	
	const size_t top_context = first_modified_line - getNumberOfFirstLine();
	const size_t bottom_context = getNumberOfLastLine(sf.getNumberOfLastLine()) - last_modified_line;
	
	const size_t to_range_count = lines_added_in_hunk + not_affected_lines + top_context + bottom_context;
	const size_t from_range_count = lines_deleted_in_hunk - lines_added_in_hunk + to_range_count;
	
	cout << "@@ -"
		<< from_range_start
		<< ','
		<< from_range_count
		<< " +"
		<< to_range_start
		<< ','
		<< to_range_count
		<< " @@"
		<< endl;
	
	vector<size_t>::const_iterator current_deleted_line = deleted_lines.begin();
	vector<size_t>::const_iterator current_added_line = added_lines.begin();
	
	for(size_t i = from_range_start; i < from_range_start + from_range_count; ++i) {
		
		bool just_line = true;
		
		if (current_deleted_line != deleted_lines.end() && *current_deleted_line == i) {
			cout << '-' << sf[i] << endl;
			current_deleted_line++;
			just_line = false;
		}
		
		if (current_added_line != added_lines.end() && *current_added_line == i) {
			
			map<size_t, size_t>::const_iterator it = replaced_lines.find(i);
			
			// It shouldn't happens because each element in
			// added_lines set is key in this map
			assert(it != replaced_lines.end());
			
			const size_t replace_lineno = it->second;
			
			const DirectiveWithCondition *dvc =
				static_cast<const DirectiveWithCondition *>(sf.getLineOfSourceCodeClass(replace_lineno));
			
			cout << '+' << dvc->toStringWithInvertedCondition() << endl;
			current_added_line++;
			just_line = false;
		}
		
		if (just_line) {
			cout << ' ' << sf[i] << endl;
		}
	}
	
}

size_t
Hunk::getNumberOfFirstDeletedLine() const {
	return (deleted_lines.empty() ? 0 : *min_element(deleted_lines.begin(), deleted_lines.end()));
}

size_t
Hunk::getNumberOfLastDeletedLine() const {
	return (deleted_lines.empty() ? 0 : *max_element(deleted_lines.begin(), deleted_lines.end()));
}

size_t
Hunk::getNumberOfFirstAddedLine() const {
	return (added_lines.empty() ? 0 : *min_element(added_lines.begin(), added_lines.end()));
}

size_t
Hunk::getNumberOfLastAddedLine() const {
	return (added_lines.empty() ? 0 : *max_element(added_lines.begin(), added_lines.end()));
}

size_t
Hunk::getNumberOfFirstLine() const {
	
	size_t first_modified_line = getNumberOfFirstModifiedLine();
	
	// for first 3 lines ranges starts from first line
	if (first_modified_line <= context) {
		return DirectSourceFile::getNumberOfFirstLine();
	}
	
	return first_modified_line - context;
}

size_t
Hunk::getNumberOfLastLine(size_t lines_in_file) const {
	
	size_t last_modified_line = getNumberOfLastModifiedLine();
	
	if (last_modified_line + context > lines_in_file) {
		return lines_in_file;
	}
	
	return last_modified_line + context;
}

size_t
Hunk::getNumberOfFirstModifiedLine() const {
	
	const size_t first_deleted_line = getNumberOfFirstDeletedLine();
	const size_t first_added_line   = getNumberOfFirstAddedLine();
	
	// Shouldn't happens because this function should called only when
	// deleted or added lines exists
	assert(first_deleted_line != 0 || first_added_line != 0);
	
	if (first_deleted_line == 0 || first_added_line == 0) {
		return max(first_deleted_line, first_added_line);
	}
	
	return min(first_deleted_line, first_added_line);
}

size_t
Hunk::getNumberOfLastModifiedLine() const {
	return max(getNumberOfLastDeletedLine(), getNumberOfLastAddedLine());
}

bool
Hunk::isLineBelongsToHunk(size_t lineno) const {
	
	const size_t first_modified_line = getNumberOfFirstModifiedLine();
	const size_t last_modified_line  = getNumberOfLastModifiedLine();
	
	if (lineno >= first_modified_line && lineno <= last_modified_line) {
		return true;
	
	} else if (lineno <= first_modified_line) {
		return (DirectSourceFile::getNumberOfNextLine(lineno)
				>= (first_modified_line + (context * 2)));
	
	} else if (lineno >= last_modified_line) {
		return (DirectSourceFile::getNumberOfPreviousLine(lineno)
				<= (last_modified_line + (context * 2)));
	}
	
	// Shouldn't happens because we already test all possible variants
	assert(0);
	
	return false;
}

#ifndef ENABLE_BOOST
bool
Hunk::isLineBelongsToHunk(Hunk hunk, size_t lineno) {
	return hunk.isLineBelongsToHunk(lineno);
}
#endif

int
Hunk::getOffsetCreatedByHunk() const {
	return added_lines.size() - deleted_lines.size();
}

