/*
 * Copyright (C) 2007-2009 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

/**
 * @class LineOfSourceCode
 * Line of source code.
 * Class used for presenting lines in file with source code.
 **/

#ifndef LINEOFSOURCECODE_HH
#define LINEOFSOURCECODE_HH

#include <string>
#include <ostream>

typedef enum LineType {
	CODE,		// line contains code
	COMMENT,	// line is comment
	BLANK,		// line is empty
	USRHDR,		// line is #include "..." directive
	SYSHDR,		// line is #include <...> directive
	UNKHDR,		// line is #include DIR directive
	IF,			// line is #if directive
	IFDEF,		// line is #ifdef directive
	IFNDEF,		// line is #ifndef directive
	ELIF,		// line is #elif directive
	ELSE,		// line is #else directive
	ENDIF,		// line is #endif directive
	DEFINE,		// line is #define directive
	UNDEF,		// line is #undef directive
	UNKDIR		// line is one of unsupported directives
} LineType;

class LineOfSourceCode {
friend std::ostream &operator<<(std::ostream &, const LineOfSourceCode &);

public:
	LineOfSourceCode(std::string line, size_t lineNumber, LineType lineType)
		: sourceLine(line), type(lineType), lineno(lineNumber) {}
	virtual ~LineOfSourceCode() {}

	bool isUserInclude() const		{return getType() == USRHDR;}
	bool isSystemInclude() const	{return getType() == SYSHDR;}
	
	bool isInclude() const			{return getType() == SYSHDR || getType() == USRHDR || getType() == UNKHDR;}

	LineType getType() const		{return type;}

	size_t getLineNumber() const {return lineno;}

protected:
	const std::string sourceLine;	// content of line (used only for patch producing)
	LineType type;			// type of line

private:
	size_t lineno;			// number of line
};

// Uses only for patch producing
inline std::ostream &
operator<<(std::ostream &os, const LineOfSourceCode &losc) {
	os << losc.sourceLine;
	return os;
}

#endif // !LINEOFSOURCECODE_HH

