/*
 * Copyright (C) 2008-2009 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Patch.hh"
#include "Hunk.hh"
#include "DirectSourceFile.hh"
#include "system_error.hh"

#include <algorithm>	// for find_if(), for_each()
#include <iostream>		// for cerr, cout, endl

#include <cassert>	// for assert()
#include <cerrno>	// for errno
#include <cstring>	// for strerror() and strcmp() (during !NDEBUG)
#include <ctime>	// for time(), localtime() and strftime()
#include <sstream>	// for ostringstream

#ifndef NDEBUG
#include <cstdlib>	// for getenv()
#define TEST_EXCEPTION_ENV_VARIABLE "HLINT_FUNC_FAIL"
#endif

#ifdef ENABLE_BOOST
#include <boost/bind.hpp>
#include <boost/scoped_array.hpp>
#else
#include <functional>	// for bind2nd(), ptr_fun(), bind1st()
#endif // !ENABLE_BOOST

// for stat()
#include <sys/types.h>
#include <sys/stat.h>
#ifndef _WIN32
// this header not needed for stat() on Windows
#include <unistd.h>
#endif

#ifdef ENABLE_BOOST
using boost::bind;
using boost::scoped_array;
#else
using std::bind1st;
using std::bind2nd;
using std::ptr_fun;
#endif

using std::cerr;
using std::cout;
using std::endl;
using std::find_if;
using std::map;
using std::ostringstream;
using std::set;
using std::string;
using std::vector;

#ifndef _WIN32
// "%F %T %z" is equivalent to "%Y-%m-%d %H:%M:%S %z"
#define STRFTIME_FORMAT "%F %T %z"
#else
// Don't use %F and %T because VC++2005 don't understand it.
// Also don't use %z because in Windows it returns time zone name.
#define STRFTIME_FORMAT "%Y-%m-%d %H:%M:%S +0000"
#endif // !_WIN32

/**
 * Convert date in unix time format to string with date.
 *
 * Result string has format specified in \ref STRFTIME_FORMAT
 * definition: <tt>"YYYY-MM-DD HH:MM:SS +0000"</tt>.
 *
 * @param[in] time date in unix time format
 * @return string with date
 * @exception system_error when localtime(3) or strftime(3) fails
 **/
string
Patch::convertTimeToDate(const time_t *time) {
	
	assert(time != NULL);
	
	struct tm *tmp = localtime(time);
	if (tmp == NULL) {
		throw system_error(strerror(errno), "localtime(3)");
	}
	
#ifndef NDEBUG
	char *var = getenv(TEST_EXCEPTION_ENV_VARIABLE);
	if (var != NULL && strcmp(var, "localtime") == 0) {
		throw system_error(strerror(EOVERFLOW), "localtime(3)");
	}
#endif
	
	// buffer for string + terminating null byte
	const size_t SIZE = sizeof("YYYY-MM-DD HH:MM:SS +0000");
	
#ifndef ENABLE_BOOST
	char *buff = new char[SIZE]();
	
	size_t ret = strftime(buff, SIZE, STRFTIME_FORMAT, tmp);
#else
	scoped_array<char> buff(new char[SIZE]());
	
	size_t ret = strftime(buff.get(), SIZE, STRFTIME_FORMAT, tmp);
#endif
	
#ifndef NDEBUG
	if (var != NULL && strcmp(var, "strftime") == 0) {
#ifndef ENABLE_BOOST
		delete [] buff;
#endif
		throw system_error("Cannot format date and time", "strftime(3)");
	}
#endif
	
	if (ret == 0) {
#ifndef ENABLE_BOOST
		delete [] buff;
#endif
		throw system_error("Cannot format date and time", "strftime(3)");
	}
	
	// From man strftime(3):
	// The strftime() function returns the number of characters placed
	// in  the array s, not including the terminating null byte.
	assert(ret == SIZE-1);
	
#ifndef ENABLE_BOOST
	string date(buff);
	
	delete [] buff;
#else
	string date(buff.get());
#endif
	
	return date;
}

#undef STRFTIME_FORMAT

/**
 * Get file modification date.
 *
 * Result string has format specified in \ref STRFTIME_FORMAT
 * definition: <tt>"YYYY-MM-DD HH:MM:SS +0000"</tt>.
 *
 * @param[in] fileName file for which we want to get modification date
 * @return string with date
 * @exception system_error when stat(2) fails
 **/
string
Patch::getFileModificationDate(const std::string &fileName) {
	
	int ret;
	struct stat fileStat;
	
	// get file information (we need only st_mtime member)
	ret = stat(fileName.c_str(), &fileStat);
	if (ret == -1) {
		throw system_error(strerror(errno), "stat(2)");
	}

#ifndef NDEBUG
	char *var = getenv(TEST_EXCEPTION_ENV_VARIABLE);
	if (var != NULL && strcmp(var, "stat") == 0) {
		throw system_error(strerror(ENOENT), "stat(2)");
	}
#endif
	
	return convertTimeToDate(&fileStat.st_mtime);
}

/**
 * Get current date.
 *
 * Result string has format specified in \ref STRFTIME_FORMAT
 * definition: <tt>"YYYY-MM-DD HH:MM:SS +0000"</tt>.
 *
 * @return string with date
 * @exception system_error when time(2) fails
 **/
string
Patch::getCurrentDate() {
	
	time_t t = time(NULL);
	if (t == static_cast<time_t>(-1)) {
		throw system_error(strerror(errno), "time(2)");
	}
	
#ifndef NDEBUG
	char *var = getenv(TEST_EXCEPTION_ENV_VARIABLE);
	if (var != NULL && strcmp(var, "time") == 0) {
		throw system_error(strerror(EFAULT), "time(2)");
	}
#endif
	
	return convertTimeToDate(&t);
}

string
Patch::getHeader(const std::string &fileName) {
	
	ostringstream header;
	
	header << "--- " << fileName << ".orig" << '\t' << getFileModificationDate(fileName) << endl;
	header << "+++ " << fileName << '\t' << getCurrentDate() << endl;
	
	return header.str();
}

void
Patch::printBanner() {
#ifndef PACKAGE_STRING
// this situation shouldn't happens because autotools define it for us
#error "PACKAGE_STRING wasn't defined!"
#else
	cout << "\n    Patch was generated by " << PACKAGE_STRING << "\n" << endl;
#endif
}

void
Patch::addLineToHunkAsRemoved(std::vector<Hunk> *hunks, size_t lineno) {
	
	vector<Hunk>::iterator it;
	
	// find in existing hunks
	it = find_if(hunks->begin(),
			hunks->end(),
#ifndef ENABLE_BOOST
			bind2nd(ptr_fun(Hunk::isLineBelongsToHunk), lineno)
#else
			bind(&Hunk::isLineBelongsToHunk, _1, lineno)
#endif
			);
	
	if (it != hunks->end()) {
		it->pushLineForDeletion(lineno);
	
	// create new if it doesn't exists
	} else {
		hunks->push_back(Hunk(lineno));
	}
}

void
Patch::addLineToHunkAsAdded(std::vector<Hunk> *hunks, size_t lineno) {
	
	vector<Hunk>::iterator it;
	
	// find in existing hunks
	it = find_if(hunks->begin(),
			hunks->end(),
#ifndef ENABLE_BOOST
			bind2nd(ptr_fun(Hunk::isLineBelongsToHunk), lineno)
#else
			bind(&Hunk::isLineBelongsToHunk, _1, lineno)
#endif
			);
	
	if (it != hunks->end()) {
		it->pushLineForAddition(lineno);
	
	// create new if it doesn't exists
	} else {
		hunks->push_back(Hunk(lineno, false));
	}
}

void
Patch::produce(const DirectSourceFile &sf,
		const std::set<size_t> &deleted_lines_list,
		const std::set<size_t> &added_lines_list,
		const std::map<size_t, size_t> &replaced_lines,
		bool print_banner) {
	
	// if duplicates wasn't found or we shouldn't add something
	if (deleted_lines_list.empty() && added_lines_list.empty()) {
		// nothing interesting found for producing patch
		return;
	}
	
	// construct header before printing banner:
	// if some of system function fails we don't need show banner, only error
	// message.
	string header = getHeader(sf.getFileName());
	
	if (print_banner) {
		// show our blurb
		printBanner();
	}
	
	cout << header;
	
	vector<Hunk> hunks;
	
	for_each(deleted_lines_list.begin(),
			deleted_lines_list.end(),
#ifndef ENABLE_BOOST
			bind1st(ptr_fun(Patch::addLineToHunkAsRemoved), &hunks)
#else
			bind(&Patch::addLineToHunkAsRemoved, &hunks, _1)
#endif
			);
	
	for_each(added_lines_list.begin(),
			added_lines_list.end(),
#ifndef ENABLE_BOOST
			bind1st(ptr_fun(Patch::addLineToHunkAsAdded), &hunks)
#else
			bind(&Patch::addLineToHunkAsAdded, &hunks, _1)
#endif
			);
	
	// for two and more hunks we should known how many lines was
	// removed/addded in all previous hunks
	int lines_offset = 0;
	
	for (vector<Hunk>::const_iterator hunk = hunks.begin();
			hunk != hunks.end();
			++hunk) {
		
		hunk->printHunk(sf, replaced_lines, lines_offset);
		
		lines_offset += hunk->getOffsetCreatedByHunk();
	}
}

