/*
 * Copyright (C) 2007-2009 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DirectSourceFile.hh"	// for DirectSourceFile class
#include "syntax_error.hh"		// for syntax_error exception
#include "system_error.hh"		// for system_error exception

#include <iostream>	// for cerr, endl
#include <cassert>	// for assert()
#include <cstdlib>	// for EXIT_* constant
#include <cstring>	// for strlen()

#ifdef ENABLE_BOOST
#include <boost/foreach.hpp>
#endif

using std::cerr;
using std::endl;
using std::runtime_error;
using std::vector;

int main(int argc, char *argv[]) {

	bool nflag = false;	// print line numbers
	bool pflag = false;	// produce patch
	bool rflag = false;	// recursive mode
	bool sflag = false;	// don't print banner at top of patch

	vector<const char *> files;

	if (argc < 2) {
		cerr << "Usage: hlint [-rn] file" << endl;
		cerr << "       hlint -p [-s] file..." << endl;
		return EXIT_FAILURE;
	}

	// The reason of why we not uses getopt() here is because Windows doesn't
	// have this function
	for (int i = 1; i < argc; ++i) {
		if (argv[i][0] == '-') {
			
			for (int j = 1; argv[i][j] != '\0'; ++j) {
				switch (argv[i][j]) {
					
					case 'n':
						nflag = true;
						break;
					
					case 'p':
						pflag = true;
						break;
					
					case 'r':
						rflag = true;
						break;
					
					case 's':
						sflag = true;
						break;
					
					default:
						cerr << "Unknown option: -" << argv[i][j] << endl;
						return EXIT_FAILURE;
				}
			}
		
		} else {
			files.push_back(argv[i]);
		}
	}
	
	if (files.empty()) {
	    cerr << "You should specify source file!" << endl;
	    return EXIT_FAILURE;
	
	} else if (files.size() > 1 && !pflag) {
		cerr << "Only one file should be specified!" << endl;
		return EXIT_FAILURE;
	}
	
	if (pflag && (rflag || nflag)) {
		cerr << "Option -p conflicts with -r and -n options!" << endl;
		return EXIT_FAILURE;
	}
	
	if (sflag && !pflag) {
		cerr << "Option -s required -p!"<< endl;
		return EXIT_FAILURE;
	}
	
#ifndef ENABLE_BOOST
	for (vector<const char *>::const_iterator it = files.begin();
		it != files.end();
		++it) {
	
		const char *file = *it;
#else
	foreach (const char *file, files) {
#endif
		try {
			
			DirectSourceFile sf(file, rflag);
			
			if (pflag) {
				
				// -p option conflicts with -r and -n and program should was
				// exit if they was mixed together
				assert(pflag == true && rflag == false);
				assert(pflag == true && nflag == false);
				
				sf.producePatch(!sflag);
				
				// print our blurb only once, at top of patch
				if (!sflag) {
					sflag = true;
				}
				
			} else {
				sf.printHeaderTree(nflag);
			}
			
		} catch (const syntax_error &e) {
			cerr << file << ": syntax error at line " << e.where() << ": " << e.what() << endl;
			return EXIT_FAILURE;
		
		} catch (const system_error &e) {
			cerr << "hlint exited with status " << EXIT_FAILURE
				<< " due to error from system function " << e.who() << ":" << endl
				<< e.what() << endl;
			return EXIT_FAILURE;
		
		} catch (const runtime_error &e) {
			// if what() returns empty string it means then we already print error
			// message
			if (strlen(e.what()) != 0) {
				cerr << e.what() << endl;
			}
			return EXIT_FAILURE;
		}
	}
	
	
	return EXIT_SUCCESS;
}
