#!/bin/sh

SRCDIR="${HLINT_SRCDIR:-.}"
BUILDDIR="${HLINT_BUILDDIR:-.}"
HLINT_PATH="${HLINT_PATH:-../src/hlint}"

PASSES=0
FAILS=0
FAILS_UNDER_VALGRIND=0
FAILS_AS_PLANNED=0
PASSED_NOT_PLANNED=0

# hlint version (used in replace_version() function)
VERSION='0.4'

# by default -f (exit on first failed test) is disabled
EXIT_ON_FIRST_FAIL=

# by default -s (print short output) is disabled
BE_SILENT=

# by default -b (don't print bottom summary) is disabled
DONT_PRINT_SUMMARY=

# by default -c (check with valgrind) is disabled
RUN_UNDER_VALGRIND=
VALGRIND_LOG='valgrind.out'

# for holds test numbers
TESTLIST=

TEST_FILE="$BUILDDIR/test.list"

usage() {
	cat <<EOF

Usage: $0 [-h] [-b] [-c] [-f] [-s] [testno...]

  -h  print this help message and exit
  -b  don't print summary of tests at bottom
  -c  run tests under valgrind(1) for detection memory errors/leaks
  -f  immediately exit when one of tests fails
  -s  when test fails print only its status (without details)

EOF
	exit
}

# parse command line options
# TODO: use getopt()/getopts() instead
if [ $# -gt 0 ]; then
	for opt in $@; do
		
		if [ "$opt" = '-b' ]; then
			DONT_PRINT_SUMMARY=yes
			shift
		
		elif [ "$opt" = '-c' ]; then
			RUN_UNDER_VALGRIND=yes
			shift
		
		elif [ "$opt" = '-f' ]; then
			EXIT_ON_FIRST_FAIL=yes
			shift
		
		elif [ "$opt" = '-h' ]; then
			usage
		
		elif [ "$opt" = '-s' ]; then
			BE_SILENT=yes
			shift
		
		elif printf '%s\n' "$1" | grep -q '^-'; then
			echo "Warning: unknown option $opt. Skipped." >&2
			shift
		
		elif ! printf '%s\n' "$1" | grep -xq '[[:digit:]]\+'; then
			echo "Wrong number of test ($1). Should be a digit! Skipped." >&2
			shift
		else
			TESTLIST="$TESTLIST $1"
			shift
		fi
	done
fi

test_failed() {
	printf '\033[31mfailed\033[0m\n'
}

test_failed_planned() {
	FAILS_AS_PLANNED=`expr $FAILS_AS_PLANNED + 1`
	printf '\033[33mok\033[0m\n'
}

test_passed() {
	printf '\033[32mok\033[0m\n'
}

# return 0: no errors
# return 1: errors detected
is_valgrind_found_errors() {
	if [ ! -f "$VALGRIND_LOG" ]; then
		printf '%s: File %s not found!\n' "$FUNCNAME" "$VALGRIND_LOG" >&2
		return 1
	fi
	
	FOUND_LEAKS=
	FOUND_ERRORS=
	
	grep -qs 'LEAK SUMMARY:' "$VALGRIND_LOG"
	if [ $? -eq 0 ]; then
		FOUND_LEAKS=yes
	fi
	
	grep -qs 'ERROR SUMMARY: 0 errors from 0 contexts' "$VALGRIND_LOG"
	if [ $? -ne 0 ]; then
		FOUND_ERRORS=yes
	fi
	
	if [ "$FOUND_LEAKS" = "yes" -o "$FOUND_ERRORS" = "yes" ]; then
		FAILS_UNDER_VALGRIND=`expr $FAILS_UNDER_VALGRIND + 1`
		return 1
	fi
	
	return 0
}

print_valgrind_stat() {
	if [ ! -f "$VALGRIND_LOG" ]; then
		printf '%s: File %s not found!\n' "$FUNCNAME" "$VALGRIND_LOG" >&2
		return 1
	fi
	
	if [ -z "$BE_SILENT" ]; then
		{
			grep -qs 'ERROR SUMMARY: 0 errors from 0 contexts' "$VALGRIND_LOG" ||
				grep 'ERROR SUMMARY:' "$VALGRIND_LOG"
			grep -A4 'LEAK SUMMARY:' "$VALGRIND_LOG"
		} | sed 's,^==[[:digit:]]\+==,   ,'
	fi
}

compare_files() {
	if [ $# -ne 2 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	ORIG=$1
	MODIF=$2
	
	if [ ! -r "$ORIG" ]; then
		test_failed
		echo "Fatal error: original file ($ORIG) not found!" >&2
		exit 1
	fi
	
	cmp -s $ORIG $MODIF
	
	return $?
}

# Usage:
# do_test number description [patch|fail]
do_test() {
	if [ $# -lt 2 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	NO="$1"
	MSG="$2"
	shift 2
	
	TESTNO=`printf '%03d\n' $NO`
	
	IS_PATCH=
	IS_SHOULD_FAILS=
	
	while [ $# -gt 0 ]; do
		if [ "$1" = 'patch' ]; then
			IS_PATCH='yes'
			shift
		elif [ "$1" = 'fail' ]; then
			IS_SHOULD_FAILS='yes'
			shift
		else
			echo "$FUNCNAME: unknown argument: $1" >&2
			return 1
		fi
	done
	
	if [ -n "$IS_PATCH" ]; then
		replace_date $TESTNO
		replace_version $TESTNO
	fi
	
	printf '%3d Test %s... ' "$NO" "$MSG"
	
	compare_files "$SRCDIR/$TESTNO.ok" ${TESTNO}.out
	if [ $? -ne 0 ]; then
		if [ -n "$IS_SHOULD_FAILS" ]; then
			test_failed_planned
		else
			test_failed
			FAILS=`expr $FAILS + 1`
			if [ -z "$BE_SILENT" ]; then
				diff -u "$SRCDIR/$TESTNO.ok" ${TESTNO}.out
			fi
		fi
		
		rm -f ${TESTNO}.out
		
		if [ -n "$RUN_UNDER_VALGRIND" ]; then
			is_valgrind_found_errors
			if [ $? -ne 0 ]; then
				
				print_valgrind_stat
				
				if [ -n "$EXIT_ON_FIRST_FAIL" -a -z "$IS_SHOULD_FAILS" ]; then
					rm -f "$VALGRIND_LOG"
					exit
				fi
			fi
			
			rm -f "$VALGRIND_LOG"
		fi
		
		if [ -n "$EXIT_ON_FIRST_FAIL" -a -z "$IS_SHOULD_FAILS" ]; then
			exit
		fi
		
		return
	fi
	
	if [ -n "$IS_SHOULD_FAILS" ]; then
		PASSED_NOT_PLANNED=`expr $PASSED_NOT_PLANNED + 1`
	fi
	
	rm -f ${TESTNO}.out
	
	if [ -n "$RUN_UNDER_VALGRIND" ]; then
		is_valgrind_found_errors
		
		if [ $? -ne 0 ]; then
			test_failed
			print_valgrind_stat
			rm -f "$VALGRIND_LOG"
			if [ -n "$EXIT_ON_FIRST_FAIL" -a -z "$IS_SHOULD_FAILS" ]; then
				rm -f "$VALGRIND_LOG"
				exit
			fi
			return
		fi
		
		rm -f "$VALGRIND_LOG"
	fi
	
	PASSES=`expr $PASSES + 1`
	test_passed
}

run_hlint() {
	if [ $# -lt 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	NO=
	LEN=`printf '%s' "$1" | wc -m`
	if [ $LEN -le 2 ]; then
		NO="`printf '%03d\n' $1`"
	else
		NO="$1"
	fi
	shift
	
	RUN_HLINT_WITHOUT_FILE=
	for i in 1 10 12; do
		if [ $i -eq `basename $NO` ]; then
			RUN_HLINT_WITHOUT_FILE=yes
			break
		fi
	done
	
	OUT_FILE="`basename $NO`.out"
	
	if [ -z "$RUN_HLINT_WITHOUT_FILE" ]; then
		$HLINT "$@" "$SRCDIR/$NO.in" >"$OUT_FILE" 2>&1
	else
		$HLINT "$@" >"$OUT_FILE" 2>&1
	fi
	
	QUOTED_SRCDIR="`printf '%s\n' "$SRCDIR" | sed 's|\.|\\\.|g'`"
	perl -i -pe "s|$QUOTED_SRCDIR/||g" "$OUT_FILE"
	
}

# Replace current date in file specified by number to many X symbols.
replace_date() {
	if [ $# -ne 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	TESTNO="$1"
	
	perl -i -pe 's|\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2} ([-+])\d{4}|XXXX-XX-XX XX:XX:XX \1XXXX|' \
		$TESTNO.out
}

# Replace current version of hlint in file specified by number to X
# symbols.
replace_version() {
	if [ $# -ne 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	fi
	
	TESTNO="$1"
	
	perl -i -pe "s|hlint $VERSION|hlint X.X|" $TESTNO.out
}

execute_test_by_number() {
	
	if [ $# -ne 1 ]; then
		echo "$FUNCNAME: insufficient arguments!" >&2
		return 1
	
	elif [ -z "$1" ]; then
		echo "$FUNCNAME: first argument shouldn't be empty!" >&2
		return 1
	fi
	
	TESTNO="$1"
	LINE=`grep -v '^#' "$TEST_FILE" | sed -n "${TESTNO}p"`
	
	DESC=`echo "$LINE" | awk -F'|' '{print $2}'`
	OPTS=`echo "$LINE" | awk -F'|' '{print $3}' | sed "s|@SRCDIR@|$SRCDIR|g"`
	FLAGS=`echo "$LINE" | awk -F'|' '{print $4}'`
	ENV=`echo "$LINE" | awk -F'|' '{print $5}'`
	
	if [ -n "$ENV" ]; then
		export HLINT_FUNC_FAIL="$ENV"
	fi
	
	run_hlint "$TESTNO" $OPTS
	
	if [ -n "$ENV" ]; then
		unset HLINT_FUNC_FAIL
	fi
	
	do_test "$TESTNO" "$DESC" $FLAGS
}

execute_all_tests() {
	
	while IFS='|' read TESTNO DESC OPTS FLAGS ENV; do
		echo "$TESTNO" | grep -qs '^#'
		if [ $? -eq 0 ]; then
			continue
		fi
		
		if [ -n "$ENV" ]; then
			export HLINT_FUNC_FAIL="$ENV"
		fi
		
		run_hlint "$TESTNO" `printf '%s\n' "$OPTS" | sed "s|@SRCDIR@|$SRCDIR|g"`
		
		if [ -n "$ENV" ]; then
			unset HLINT_FUNC_FAIL
		fi
		
		do_test "$TESTNO" "$DESC" $FLAGS
	done <"$TEST_FILE"
}

if [ ! -x "$HLINT_PATH" ]; then
	echo "File $HLINT_PATH not found! Please run 'make' before start tests!" >&2
	exit 1
fi

HLINT=
if [ -z "$RUN_UNDER_VALGRIND" ]; then
	HLINT="$HLINT_PATH"
else
	VALGRIND_PATH="`which valgrind`"
	if [ -n "$VALGRIND_PATH" ]; then
		unset VALGRIND_OPTS
		# NOTE:
		# If you have valgrind < 3.3.0 then replace --log-file to
		# --log-file-exactly option
		HLINT="$VALGRIND_PATH -q -v --log-file=$VALGRIND_LOG $HLINT_PATH"
	else
		echo 'Option -c specified but valgrind program not found!' >&2
		exit 1
	fi
fi

if [ -z "$TESTLIST" ]; then
	execute_all_tests
else
	for i in $TESTLIST; do
		execute_test_by_number "$i"
	done
fi

if [ -z "$DONT_PRINT_SUMMARY" ]; then
	echo
	printf '    Tests passes:\t%2d (fails as planned: %d, unexpectedly passes: %d)\n' \
				$PASSES $FAILS_AS_PLANNED $PASSED_NOT_PLANNED
	if [ -n "$RUN_UNDER_VALGRIND" ]; then
		VALGRIND_STAT=" (under valgrind: ${FAILS_UNDER_VALGRIND})"
	fi
	printf '    Tests failures:\t%2d%s\n' $FAILS "${VALGRIND_STAT:-}"
	echo
fi

if [ $FAILS -gt 0 ] || [ -n "$RUN_UNDER_VALGRIND" -a $FAILS_UNDER_VALGRIND -gt 0 ]; then
	exit 1
fi

