/*
 * Copyright (C) 2008-2009 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Algorithm.hh"
#include "SourceFile.hh"
#include "Header.hh"
#include "DirectiveWithCondition.hh"

#include <algorithm>	// for sort(), unique(), remove_if()
#include <cassert>		// for assert()
#include <map>			// for multimap class
#include <set>			// for set class
#include <stack>		// for stack adapter

#ifdef ENABLE_BOOST
#include <boost/lambda/lambda.hpp>
#include <boost/foreach.hpp>
using boost::lambda::_1;
#else
#include <functional>	// for bind2nd(), greater_equal()
using std::bind2nd;
using std::greater_equal;
#endif

using std::unique;
using std::make_pair;
using std::multimap;
using std::remove_if;
using std::set;
using std::sort;
using std::stack;
using std::string;
using std::vector;

namespace {

bool not_equal_types(const Algorithm::DuplPair &pair) {
	return pair.first->getType() != pair.second->getType();
}

bool lessByLineNumberForDrtvPairs(const Algorithm::DrtvPair &lhs, const Algorithm::DrtvPair &rhs) {
	return lhs.first->getLineNumber() < rhs.first->getLineNumber();
}

} // end of anonymous namespace


/**
 * Extract lines with \#include directive from source file.
 *
 * @param[in] sf source file
 * @return vector with pointer to lines
 **/
vector<CHdrPtr>
Algorithm::getHeaders(const SourceFile *sf) {
	
	/*
	 * 1. for each line in source file:
	 *    a) if this line is #include directive then save it to result
	 *       vector
	 **/
	
	// Shouldn't happens because we always operate with really existing
	// SourceFile class
	assert(sf != NULL);
	
	vector<CHdrPtr> result;
	
	for (size_t lineno = sf->getNumberOfFirstLine(); lineno <= sf->getNumberOfLastLine(); ++lineno) {
		
		const LineOfSourceCode *losc = sf->getLineOfSourceCodeClass(lineno);
		
		// Shouldn't happens because we iterate through file line by line
		assert(lineno == losc->getLineNumber());
		
		if (losc->isInclude()) {
			result.push_back(static_cast<CHdrPtr>(losc));
		}
		
	}
	
	return result;
}


/**
 * Get pairs of original line with \#include directive and its duplicate.
 *
 * @param[in] headers_list set of line numbers with \#include directive
 * @return vector with pairs of pointers to original line and its duplicate
 *
 * @see getHeaders()
 **/
vector<Algorithm::DuplPair>
Algorithm::getDuplicatedHeadersPairs(std::vector<CHdrPtr> headers_list) {
	
	/*
	 * 1. for each header from source file:
	 *    a) save header name to list with unique header names
	 *    b) save header name and pointer to header to header map
	 * 2. for each name from unique header names:
	 *    a) search header in header map
	 *    b) if header not unique then save pointers to original line
	 *       and to duplicated line to result vector
	 **/
	
	vector<DuplPair> result;
	
	set<string> header_names;
	
	typedef multimap<string, CHdrPtr> HdrLabelsMap;
	HdrLabelsMap headers_map;
	
#ifndef ENABLE_BOOST
	for (vector<CHdrPtr>::const_iterator cit = headers_list.begin();
		cit != headers_list.end();
		++cit) {
		
		CHdrPtr hdr = *cit;
#else
	foreach(CHdrPtr hdr, headers_list) {
#endif
		
		// Shouldn't happens because we always operate with headers
		assert(hdr->isInclude());
		
		// NOTE:
		// we use headers_map::value_type instead of make_pair() because it
		// allow building program under Sun Studio compiler
		headers_map.insert(HdrLabelsMap::value_type(hdr->getHeaderName(), hdr));
		
		header_names.insert(hdr->getHeaderName());
	}
	
#ifndef ENABLE_BOOST
	for (set<string>::const_iterator cit = header_names.begin();
		cit != header_names.end();
		++cit) {
		
		const string header_name = *cit;
#else
	foreach (const string header_name, header_names) {
#endif
		
		HdrLabelsMap::size_type counter = headers_map.count(header_name);
		
		// Shouldn't happens because we always looking for existing headers
		assert(counter != 0);
		
		// skip unique header names
		if (counter <= 1) {	// 0 when not found, 1 when found self
			continue;
		}
		
		CHdrPtr original = NULL;
		
		for (HdrLabelsMap::const_iterator it = headers_map.find(header_name);
			it != headers_map.end() && counter > 0;
			++it, --counter) {
			
			if (original == NULL) {
				// treat first header as original and memorize it
				original = it->second;
			
			} else {
				// Shouldn't happens because all lines sorted from
				// first to last
				assert(original->getLineNumber() < it->second->getLineNumber());
				
				// save all other duplicated headers
				result.push_back(make_pair(original, it->second));
			}
		}
	}
	
	return result;
}


/**
 * Exclude pairs of duplicated headers which has different types.
 *
 * @param[in] duplicated_pairs list with duplicated pairs
 * @return vector with pairs of pointers to original line and its duplicate
 *
 * @see getDuplicatedHeadersPairs()
 **/
vector<Algorithm::DuplPair>
Algorithm::excludeHeadersWithDifferentTypes(std::vector<Algorithm::DuplPair> duplicated_pairs) {
	
	/*
	 * 1. remove all pairs which has different types
	 *
	 **/
	
	vector<Algorithm::DuplPair> result(duplicated_pairs);
	
	vector<Algorithm::DuplPair>::iterator end =
		remove_if(result.begin(), result.end(), not_equal_types);
	result.erase(end, result.end());
	
	return result;
}


/**
 * Find duplicated headers which was included before specified header.
 *
 * Method gets list with duplicated headers and search all duplicates
 * for specified header which was included before (at previous lines
 * in file).
 *
 * @param[in] hdr header for which we will find
 * @param[in] duplicated_pairs list with duplicated pairs
 * @return vector with line numbers
 *
 * @see getDuplicatedHeadersPairs()
 **/
vector<size_t>
Algorithm::getEarlyIncludedHeaders(CHdrPtr hdr, std::vector<Algorithm::DuplPair> duplicated_pairs) {
	
	/*
	 * 1. save both members of duplicated pairs to result vector if
	 *    its contains duplicate for our header
	 * 2. for result vector:
	 *    a) remove duplicated line numbers
	 *    b) remove all lines which after us or at same line (self)
	 *
	 **/
	
	assert(hdr != NULL);
	assert(hdr->isInclude());
	
	vector<size_t> result;
	
	for (vector<Algorithm::DuplPair>::const_iterator cit = duplicated_pairs.begin();
			cit != duplicated_pairs.end();
			++cit) {
		if (hdr->getHeaderName() == cit->first->getHeaderName()) {
			result.push_back(cit->first->getLineNumber());
			result.push_back(cit->second->getLineNumber());
		}
	}
	
	vector<size_t>::iterator end_of_unique;
	vector<size_t>::iterator end_of_greater;
	
	// there we can use set instead of vector, but I think this way
	// should be faster
	sort(result.begin(), result.end());
	end_of_unique = unique(result.begin(), result.end());
	
	end_of_greater = remove_if(result.begin(), end_of_unique,
#ifndef ENABLE_BOOST
			bind2nd(greater_equal<size_t>(), hdr->getLineNumber())
#else
			_1 >= hdr->getLineNumber()
#endif
			);
	result.erase(end_of_greater, result.end());
	
	return result;
}


/**
 * Get pairs which marks lines where condition directives starts and ends.
 *
 * Result vector contains all condition directives (\#if, \#ifdef and \#ifndef)
 * from file which sorted in ascending order. Each pair describe one directive.
 * First element contains pointer to DirectiveWithCondition class and usually
 * used for determine directive name, type and other information. Second member
 * is number of line where directive ends (by appropriate \#endif).
 *
 * All errors which occurs during parsing saved to errors argument.
 *
 * @param[in] sf source file
 * @param[out] errors syntax errors
 * @return vector with pairs which marks lines where directive starts and ends
 *
 * @see DirectiveWithCondition
 **/
vector<Algorithm::DrtvPair>
Algorithm::getDirectivesPairs(const SourceFile *sf, std::vector<syntax_error> &errors) {
	
	/*
	 * 1. for each line in source file:
	 *    a) if has type IF/IFDEF/IFNDEF then push it to the stack
	 *    b) if has type ENDIF then try to get appropriate IF/IFDEF/IFNDEF from
	 *       stack:
	 *       - if stack is empty then save error object to vector with errors
	 *       - else get element and save pair with start/end lines to result vector
	 * 2. if stack not empty then make error object for each line in stack and
	 *    save them in vector with errors
	 * 3. sort result vector in ascending order
	 *
	 **/
	
	// Shouldn't happens because we always operate with really existing
	// SourceFile class
	assert(sf != NULL);
	
	assert(errors.empty());
	
	vector<DrtvPair> result;
	
	stack<CDrtvPtr> conditions_directives;
	
	for (size_t lineno = sf->getNumberOfFirstLine(); lineno <= sf->getNumberOfLastLine(); ++lineno) {
		
		const LineOfSourceCode *losc = sf->getLineOfSourceCodeClass(lineno);
		
		// Shouldn't happens because we iterate through file line by line
		assert(lineno == losc->getLineNumber());
		
		switch (losc->getType()) {
			
			case IF:
			case IFDEF:
			case IFNDEF:
				conditions_directives.push(static_cast<CDrtvPtr>(losc));
				break;
			
			case ENDIF:
				if (conditions_directives.empty()) {
					errors.push_back(syntax_error("#endif without #if", losc->getLineNumber()));
				} else {
					result.push_back(make_pair(conditions_directives.top(), losc->getLineNumber()));
					conditions_directives.pop();
				}
				break;
			
			default:
				break;	// do nothing
		}
	}
	
	while (!conditions_directives.empty()) {
		CDrtvPtr ptr = conditions_directives.top();
		errors.push_back(syntax_error(string("unterminated #") + ptr->getConditionName(), ptr->getLineNumber()));
		conditions_directives.pop();
	}
	
	sort(result.begin(), result.end(), lessByLineNumberForDrtvPairs);
	
	return result;
}


/**
 * Get all conditions which affects to specified line.
 *
 * Method return all condition directives around specified line.
 *
 * @param[in] condition_pairs vector with pairs of condition directives
 * @param[in] lineno line number to search for
 * @return vector with DirectiveWithCondition pointers
 *
 * @see getDirectivesPairs()
 **/
vector<Algorithm::CDrtvPtr>
Algorithm::getConditionsAffectsToLine(std::vector<DrtvPair> condition_pairs, size_t lineno) {
	
	/*
	 * 1. for each directive in vector:
	 *    a) if directives starts before specified line and ends after then line
	 *       inside directive and saved to result vector
	 **/
	
	vector<CDrtvPtr> result;
	
#ifndef ENABLE_BOOST
	for (vector<DrtvPair>::const_iterator cit = condition_pairs.begin();
		cit != condition_pairs.end();
		++cit) {
		
		const DrtvPair pair = *cit;
#else
	foreach (DrtvPair pair, condition_pairs) {
#endif
		
		CDrtvPtr drv = pair.first;
		const size_t drv_start = drv->getLineNumber();
		const size_t drv_end   = pair.second;
		
		assert(drv->getType() == IF || drv->getType() == IFDEF || drv->getType() == IFNDEF);
		
		// if line inside condition
		if (lineno > drv_start && lineno < drv_end) {
			result.push_back(drv);
		}
		
	}
	
	return result;
}

