/*
 * Copyright (C) 2007-2009 Slava Semushin <php-coder@altlinux.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include "Algorithm.hh"
#include "IndirectSourceFile.hh"
#include "LineOfSourceCode.hh"
#include "UserHeader.hh"

#include <cassert>	// for assert()
#include <iostream>	// for cout, endl
#include <sstream>	// for ostringtream
#include <stdexcept>// for runtime_error exception
#include <vector>	// for vector

using std::cout;
using std::endl;
using std::ostringstream;
using std::runtime_error;
using std::string;
using std::vector;


/**
 * @exception std::runtime_error when recursion_level equals to zero
 **/
IndirectSourceFile::IndirectSourceFile(const std::string &filename, int recursion_level)
				: SourceFile(filename), fileWasOpened(false) {
	
	assert(recursion_level >= 0);
	
	if (recursion_level == 0) {
		throw runtime_error("#include nested too deeply");
	}
	
	try {
		parseSourceFile();
	} catch (...) {
		return;
	}

	fileWasOpened = true;

	for (vector<LineOfSourceCode *>::const_iterator vec = sourceFile.begin();
		vec != sourceFile.end();
		++vec) {
		if ((*vec)->isUserInclude()) {
			UserHeader *usrhdr = static_cast<UserHeader *>(*vec);
			IndirectSourceFile *usf = new IndirectSourceFile(
					filePath + usrhdr->getHeaderName(),
					recursion_level-1);
			usrhdr->setSourceFile(usf);
		}
	}
}

void
IndirectSourceFile::printHeaderTree(bool nflag, size_t level) const {
	
	vector<CHdrPtr> all_headers =
		Algorithm::getHeaders(this);
	
	for (vector<CHdrPtr>::const_iterator vec = all_headers.begin();
			vec != all_headers.end();
			++vec) {
		
		CHdrPtr hdr = *vec;
		
		ostringstream line;
		line << string(level*2, ' ');

		if (nflag) {
			line << ':' << hdr->getLineNumber();
		}

		line << ": " << hdr->getHeaderName();

		if (hdr->isUserInclude()) {
			const UserHeader *usrhdr = static_cast<const UserHeader *>(hdr);
			if (! usrhdr->getSourceFile()->isFileExists()) {
				line << " (not found)";
			}
		}

		cout << line.str() << endl;

		if (hdr->isUserInclude()) {
			const UserHeader *usrhdr = static_cast<const UserHeader *>(hdr);
			if (usrhdr->getSourceFile()->isFileExists()) {
				usrhdr->getSourceFile()->printHeaderTree(nflag, level+1);
			}
		}

	}
}

