(define-module (alterator atlas)
  :use-module (srfi srfi-1)

  :use-module (alterator ports)
  :use-module (alterator algo)
  :use-module (alterator str)
  :use-module (alterator object)
  :use-module (alterator glob)
  :use-module (alterator context)
  :export (make-file-item
           make-view-item
           make-memory-item

           make-atlas
           load-atlas

           with-atlas))

(define *system-map-dir* "/usr/share/alterator/maps/")
(define *local-map-dir* "maps/")

(define *system-ui-dir* "/usr/share/alterator/ui")
(define *local-ui-dir* "ui")

(define (make-system-map-name name)
  (string-append *system-map-dir* name))

(define (make-local-map-name name)
  (and (fluid-ref use-local-files)
       (string-append *local-map-dir* name)))

(define (make-system-ui-name name index)
  (string-append *system-ui-dir* (->string name) index ".scm"))

(define (make-local-ui-name name index)
  (and (fluid-ref use-local-files)
       (string-append *local-ui-dir* (->string name) index ".scm")))

(define-macro (with-atlas <atlas> <instruction> . <instructions>)
  `(with-changed-context 'atlas ,<atlas> ,<instruction> . ,<instructions>))

(define (auto-search url . params)
  (let ((system-name (make-system-ui-name url ""))
        (local-name  (make-local-ui-name url ""))
        (system-idx-name (make-system-ui-name url "/index"))
        (local-idx-name  (make-local-ui-name  url "/index")))
    (with-first-readable
     (list local-name system-name local-idx-name system-idx-name)
     (lambda(file-name)
       (apply make-file-item file-name params)))))

(define (auto-list url)
  (let ((item (auto-search url)))
    (if (procedure? item)
        (list item)
        '())))

;;memory mapper item
(define (make-memory-item contents . params)
  (object
   (lambda (mappings . other-params)
     (apply make-memory-item contents (append params other-params)))
   ((params-of self) params)
   ((call-with-port self proc) (call-with-input-string contents proc))))

;;file mapper item
(define (make-file-item file-name . params)
  (object
   (lambda (mappings . other-params)
     (apply make-file-item file-name (append params other-params)))
   ((params-of self) (append params (list 'atlas:file file-name)))
   ((call-with-port self proc) (call-with-input-file/informative file-name proc))))

;;view mapper item
(define (make-view-item next-url . params)
  (object
   (lambda (mappings . other-params)
     (cond
      ((assq next-url mappings) => (lambda (item)
                                     (apply (cdr item) mappings (append params other-params))))
      (else (apply auto-search next-url (append params other-params)))))
   ((params-of self) (error "cannot get params for transit node"))
   ((next-url-of self) next-url)
   ((call-with-port self proc) (error "cannot get port for transit node"))))

(define (a-append alist name value)
  (append1 alist (cons name value)))

(define (valid-url? url)
  (lambda(item)
    (and (eq? (car item) url) (cdr item))))

;;mapper itself
(define (make-atlas mappings)
  (define-operation get-maps)
  (define-operation join-maps)

  (object
   (lambda (url . params)
     (cond
      ((assq (sure-symbol url) mappings) => (lambda (item)
                                              (apply (cdr item) mappings (cons* 'atlas:url url params))))
      (else (apply auto-search url (cons* 'atlas:url url params)))))
   ((add-file self url file-name . params)
    (make-atlas (a-append mappings url (apply make-file-item file-name params))))
   ((add-view self url next-url . params)
    (make-atlas (a-append mappings url (apply make-view-item next-url params))))
   ((add-memory self url contents . params)
    (make-atlas (a-append mappings url (apply make-memory-item contents params))))
   ((get-maps self) mappings)
   ((join-maps self mapper)
    (make-atlas (append mappings (get-maps mapper))))
   ((all-matches self url)
    (let ((ans (filter-map (valid-url? url) mappings)))
      (if (pair? ans)
          ans
          (auto-list url))))
   ((add-include self filespec)
    (let* ((specs (delq #f (list (make-system-map-name filespec)
                                 (make-local-map-name filespec)))))
      (define-operation get-maps)
      (fold
       (lambda (file-name mapper)
         (join-maps mapper (load-atlas (basename file-name))))
       self
       (delete-duplicates
        (sort (append-map glob specs)
              string<?)))))))

;;convert command to single-name
(define (convert-cmd cmd)
  (if (eq? (car cmd) 'include)
      cmd
      `(,(cadr cmd) ,(car cmd) ,@(cddr cmd))))

(define (load-atlas file-name)
  (let ((system-name (make-system-map-name file-name))
        (local-name (make-local-map-name file-name)))
    (with-first-readable
     (list local-name system-name)
     (lambda(file-name)
       (call-with-input-file/informative
        file-name
        (lambda (port)
          (let loop ((mapper (make-atlas '()))
                     (cmd (read port)))
            (if (eof-object? cmd)
                mapper
                (loop (let ((cmd (convert-cmd cmd)))
                        (define-operation add-file)
                        (define-operation add-view)
                        (define-operation add-include)
                        (define-operation add-memory)

                        (case (car cmd)
                          ((include) (add-include mapper (cadr cmd)))
                          ((file) (apply add-file mapper (cadr cmd) (caddr cmd) (cdddr cmd)))
                          ((view) (apply add-view  mapper (cadr cmd) (caddr cmd) (cdddr cmd)))
                          ((memory) (apply add-memory mapper (cadr cmd) (caddr cmd) (cdddr cmd)))
                          (else (error "unknown mapper command" cmd))))
                      (read port))))))))))
