(define-module (alterator configd)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-13)

  :use-module (alterator glob)
  :use-module (alterator str)
  :use-module (alterator algo)
  :use-module (alterator plist)
  :use-module (alterator woo)
  :use-module (alterator gettext)

  :use-module (alterator http html)
  :use-module (alterator http template)

  :use-module (alterator configd frontend)
  :use-module (alterator configd html)
  :use-module (alterator configd woo)
  
  :export (configd))

(define (other-option? x)
  (not (eq? (car x) 'value)))

(define (translate-value options translator)
  (let ((value (cond-assq 'value options)))
    (if value
        (acons 'value (translator value)
               (filter other-option? options))
        options)))

(define (translate-script langlist)
  (let ((_ (make-translator "alterator-fbi" langlist)))
    (html: "script"
           (call-with-output-string
            (lambda(out)
              (format out "var po={};~%")
              (call-with-input-file
                  "/var/www/html/fbi/messages.js.in"
                (lambda(in)
                  (let loop ((value (read in)))
                    (or (eof-object? value)
                        (begin
                          (format out "po[~S]=~S;" value (_ value))
                          (loop (read in))))))))))))

;;translate template
(define (translate-form scm url url-args template-args)
  (let* ((langlist (cond-assoc "language" url-args '("en_US")))
         (pkg (cond-assq 'translate template-args (string-append "alterator-" (base-path url))))
         (translator (make-translator pkg langlist)))
    (scm-filter
     scm
     (make-cb
      (tag: "head"
            (@ 'template-operation 'append-content)
            (translate-script langlist))
      (replace-tag: "span"
            (@ 'translate "_")
            (lambda (options content)
              (translator (apply string-append (filter string? content)))))
      (replace-tag: "input"
            (@ 'type "submit")
            (lambda (options content)
              `(input ,@(translate-value options translator) ,content)))
      (replace-tag: "input"
            (@ 'type "reset")
            (lambda (options content)
              `(input ,@(translate-value options translator) ,content)))))))

;;main workflow engine
(define (workflow-form action name url url-args template-args)
  (woo-get-option
   (woo-first-command (woo-try action
                               (string-append "workflow-" name "/" url)
                               'url-args url-args
                               'template-args template-args))
   'scm))

;; unified "post/get" processing
(define (template<->form action url url-args)
  (let* ((template-args (apply woo-template url url-args))
         (name (cond-assq 'template template-args "form"))
         (scm (workflow-form action name url url-args template-args)))
    (if (cond-assoc "async" url-args)
        scm
        (translate-form scm url url-args template-args))))

;; "get" template processing
(define (template->form url url-args)
   (template<->form  "read" ;;enforce read operation
                     url
                     url-args))

;; "post" template processing
(define (template<-form url url-args)
   (template<->form (cond-assoc "action" url-args "write")
                    url
                    url-args))

;;redirect to index backend
(define (url-for-args url)
  (if (string=? url "/") "/index" url))

(define (configd-main method url url-args . rest)
  (catch #t
    (thunk
     (cond
      ((string-ci=? method "post")
       (template<-form (url-for-args url) url-args))
      ((string-ci=? method "get")
       (template->form (url-for-args url) url-args))
      ((string-ci=? method "woo")
       (apply woo-try url-args url rest))
      (else (write '() ))))
      (html:exception url-args)))

(define (configd)
  (lambda (cmds next)
    (catch #t
      (thunk
       (with-fluids
        ((woo-gate next))
        (apply configd-main (car cmds))))
      (html:exception '() ))))
