(define-module (alterator ensign)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :use-module (srfi srfi-13)
  :use-module (alterator object)
  :use-module (alterator str)
  :use-module (alterator plist)
  :use-module (alterator algo)
  :use-module (alterator woo)
  :use-module (alterator constraints)
  
  :use-module (alterator exit-handler)
  
  :use-module (alterator ensign barnacle)
  :use-module (alterator ensign midshipman)
  :use-module (alterator ensign backend3)
  :use-module (alterator ensign ensign)
  
  :export (ensign
           ensign-gate))

;description: woobus module to operate with backends
(define *mariners* (make-cell '()))
(define *clean-interval* (make-cell (* 60 60 5)))

(define *system-backend-dir* "/usr/lib/alterator/backend")
(define *local-backend-dir* "backend")

(define (stop-mariners . ignore)
  (define-operation quit #t)
  (cell-set! *mariners* (filter
                         (lambda(x) (not (quit (cadr x))))
                         (cell-ref *mariners*))))
  
(sigaction SIGHUP stop-mariners) ;;at sighup reset
(at-exit stop-mariners); at exit reset

;;automatic cleanup of the unused mariners
(define (sweep-mariners)
  (define-operation quit #t)
  (let ((clean-interval (cell-ref *clean-interval*))
        (now (current-time)))
    (cell-set! *mariners*
               (filter
                (lambda(x)
                  (or (< (- now (cddr x)) clean-interval)
                      (not (quit (cadr x)))))
                (cell-ref *mariners*)))))

;try to load an appropriate backend
(define (load-mariner sym)
  (or (make-midshipman sym)
      (make-backend3 sym)
      (make-barnacle sym)))

(define (remember-mariner sym backend)
  (and backend
       (cell-set! *mariners*
                  (acons sym (cons backend (current-time)) (cell-ref *mariners*))))
  backend)

(define (touch-mariner x)
    (set-cdr! (cdr x) (current-time))
    (sweep-mariners)
    (cadr x))

(define (resolv-name sym)
  (cond
   ((assoc sym (cell-ref *mariners*)) => touch-mariner)
   (else (remember-mariner sym (load-mariner sym)))))

(define (run-mariner mariner action objects options)
  (define (pair-return z) (and (list? z) z))
  (run-operation mariner
                 action
                 (lambda(mariner . args)
                   (or (pair-return (apply mariner mariner args))
                       `(error ,(string-append "unknown action for backend "
                                               (car objects)))))
                 (cdr objects)
                 options))

;load low-level backend
(define (call-mariner cmd)
  (let* ((name (car cmd))
         (args (cdr cmd))
         (mariner (resolv-name (car name))))
    (cond
     ((not mariner) '(error "backend not found"))
     ((plistq 'action args) =>
      (lambda(action)
        (run-mariner mariner (sure-symbol (cdr action))
                     (car cmd)
		     (cdr cmd))))
     (else `(error "action field not found" (format #f "cmd=~S" cmd)) ))))


(define (error-answer? cmd) (and (pair? cmd) (eq? (car cmd) 'error)))
(define (quote-answer? cmd) (and (pair? cmd) (eq? (car cmd) 'quote)))
(define (list-answer? cmd)  (and (pair? cmd) (pair? (car cmd))))

(define (out-normal name options)
  (cons name options))
(define (out-error name reason)
  (out-normal (string-append "/error" name) `(reason ,reason)))

(define (prepare-name name)
  (if (pair? name)
      name
      (and (not-empty-string? name)
           (not (string-index name #\*))
           (string-cut-repeated name #\/))))

;filter: split command name, convert args to associative list
(define (prepare-cmds cmds)
  (filter-map (lambda(cmd)
                (and-let* (((pair? cmd))
                           (name (prepare-name (car cmd)))
                           ((not (null? name)))
                           ((not (member ".." name))))
                          (cons name (cdr cmd))))
              cmds))

; new-version: process commands from list and autoconvert results
(define (process-cmd cmd)
  (let ((name (string-append "/" (string-join (car cmd) "/")))
        (answer (call-mariner cmd)))
    (cond
     ((null? answer) '())
     ;;answer will be destroyed from output queue, use (quote) for really empty answers
     ((error-answer? answer) (list (out-error name (cadr answer))))
     ((quote-answer? answer) (list (cdr answer)))
     ((list-answer? answer)
      (map (lambda (x)
             (if (quote-answer? x)
                 (cdr x)
                 (out-normal (string-append name "/" (car x))
                              (cons* 'name (car x) (cdr x)))))
             answer))
     ((pair? answer) (list (out-normal name answer)))
     (else
      (list (out-error name "unsupported answer type"))))))

(define (ensign-gate cmds)
  (with-fluids ((woo-gate ensign-gate))
               (append-map process-cmd
                           (prepare-cmds cmds))))

;main entry point
(define (ensign)
  ;;load default backends
  (let ((backend (make-ensign *mariners* *clean-interval*)))
    (cell-set! *mariners*
               (acons "ensign" (cons backend (current-time)) '() )))
  (lambda(cmds next)
    (next (ensign-gate cmds))))
