;helper functions to work with cgi/fastcgi interface
(define-module (alterator http cgi)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :use-module (srfi srfi-13)

  :use-module (alterator algo)
  :use-module (alterator str)
  
  :export (get-cookie
           set-cookie
           
           get-languages
           
	   call-with-content
           put-message
	   
           content-type
	   set-content-type
           
           request-method
           remote-address
	   path-info
	   script-name
	   query-string
           http-header-variable

           decode-content
	   decode-url
           decode-multipart))

(dynamic-call "scm_init_fcgi"
	      (dynamic-link "libguile-fcgi.so"))

;;; common splitters
(define (line-trim x)
  (string-trim-both x #\space))

(define (line-param x)
  (let ((x (map line-trim (string-cut-repeated x #\=))))
    (cons (car x) (cond-cadr x))))

(define (line-split x)
  (map line-param (string-cut-repeated x #\;)))

;;;;;;; input/output

(define (call-with-content proc)
  (do ((next-input (fcgi-accept) (fcgi-accept)))
      ((not next-input) 'done)
    (proc (fcgi-get (content-length)))))

(define (put-message str)
  (fcgi-put str))

;;;;;;; variable testing

(define (request-method)
  (or (getenv "REQUEST_METHOD") "GET"))

(define (remote-address)
  (getenv "REMOTE_ADDR"))

(define (path-info)
  (or (getenv "PATH_INFO") ""))

(define (script-name)
  (or (getenv "SCRIPT_NAME") ""))

(define (query-string)
  (or (getenv "QUERY_STRING") ""))

(define (content-length)
  (string->number (or (getenv "CONTENT_LENGTH") "0")))

(define (http-variable-quote str)
  (string-quote
   (lambda(char)
     (case char
       ((#\-) "_")
       (else (string char))))
   str))

(define (http-header-variable str)
  (getenv (string-append "HTTP_"
                         (string-upcase (http-variable-quote str)))))

;;;;;; cookie read/write

(define (get-cookie)
  (filter cdr (line-split (or (getenv "HTTP_COOKIE") ""))))

(define (set-cookie lst)
  (for-each
    (lambda(x)
      (put-message (format #f "set-cookie: ~A = ~A~%" (car x) (cdr x))))
    lst))

;;;;;;; content type

(define (set-content-type type)
  (put-message (format #f "content-type: ~A~%~%" type)))

(define (content-type)
  (line-split (or (getenv "CONTENT_TYPE") "text/plain")))

;;;;;;; extract languages

;;extract q from parameters
;;FIXME: use line-split
(define (get-q params)
  (let ((result
         (find (lambda(param)
                 (and (= (length param) 2)
                      (string=? (car param) "q")))
               (map (lambda(item)
                      (string-cut item #\=))
                    params))))
    (if result
        (string->number (cadr result))
        1)))

(define (compare-q x y)
  (> (cdr x) (cdr y)))

;;FIXME: use line-split
(define (get-languages)
  (let ((langlist (getenv "HTTP_ACCEPT_LANGUAGE")))
    (and (string? langlist)
        (map car
             (stable-sort
              (map (lambda (item)
                     (let ((attr (string-cut item #\;)))
                       (cons (car attr) (get-q (cdr attr)))))
                   (string-cut (string-delete langlist #\space) #\,))
              compare-q)) )))

;;; url decoding

(define (hex->char x y)
    (integer->char
     (string->number (string x y) 16)))

(define (decode-url-chars s)
  (let ((s (string->list s)))
    (list->string
     (let loop ((s s)
                (result '()))
       (if (null? s)
           (reverse result)
           (let ((a (car s)) (d (cdr s)))
             (case a
               ((#\+)
                (loop d (cons #\space result)))
               ((#\%)
                (loop (cddr d)
                      (cons (hex->char (car d) (cadr d)) result)))
               (else (loop d (cons a result))))))))))

(define (decode-url-component x)
  (let ((x (string-cut x #\=)))
    (if (< (length x) 2)
        (error "wrong url component" x)
        (cons (string-delete (decode-url-chars (car x)) #\cr)
              (string-delete (decode-url-chars (cadr x)) #\cr)))))

(define (decode-url s)
  (if (string-null? s)
      '()
      (map decode-url-component (string-cut s #\&))))

;;; multipart decoding

;;returns : start-index,last-index
(define (make-next-part s boundary len)
  (lambda (start)
    (let loop ((idx (string-contains s boundary start)))
      (and idx
           (let ((last-idx (+ idx len)))
             (cond
              ;;intermediate boundary
              ((string-contains s (string #\newline) last-idx)
               (cons
                (substring s start idx)
                (+ last-idx 1)))
              ;;final boundary
              ((string-contains s (string #\- #\- #\newline) last-idx)
               (cons (substing s start idx)
                     (+ last-idx 3)))
              (else (loop (string-contains s boundary last-idx)))))))))

(define (multipart-split s boundary)
  (and boundary
       (let* ((boundary-len (string-length boundary))
              (next-part (make-next-part s boundary boundary-len)))
         (let loop ((result '())
                    (start (+ boundary-len 1))) ;;skip first boundary with #\newline final character
           (let ((next (next-part start)))
             (if (pair? next)
                 (loop
                  (cons (car next) result)
                  (cdr next))
                 (reverse result)))))))

(define (multipart-boundary type)
  (string-append "--" (cond-assoc "boundary" (cdr type) "")))

(define (multipart-name s)
  (and-let* ((idx (string-contains-ci s "Content-Disposition:"))
             (last-idx (string-index s #\newline idx)))
            (string-trim-both 
             (cond-assoc "name"
                         (line-split (substring s idx last-idx))
                         "unknown")
             #\")))

;;TODO: recognize content-type of the fragment
(define (multipart-data s)
  (and-let* ((idx (string-contains s (string #\newline #\newline))))
            (string-drop-right (substring s (+ idx 2)) 1)))

(define (decode-multipart s type)
  (map (lambda(x)
         (cons (multipart-name x)
               (multipart-data x)))
       (multipart-split (string-delete s #\cr) (multipart-boundary type))))

;;; common decoding wrapper
(define (decode-content s)
  (if (string-null? s)
      '()
      (let* ((type (content-type))
             (type-name (caar (content-type))))
        (cond
         ((string-ci=? type-name "application/x-www-form-urlencoded")
          (decode-url s))
         ((string-ci=? type-name "multipart/form-data")
          (decode-multipart s type))
         (else (error "Unsupported content type"))))))
