;helper functions to work with cgi/fastcgi interface
(define-module (alterator http template)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-11)
  :use-module (srfi srfi-13)
  :use-module (alterator plist)
  :use-module (alterator algo)
  :use-module (alterator http html)
  :export (;;low level
           file->scm
           scm-filter
           scm->out
           make-cb
           ;;high level
           template
           tag:
	   replace-tag:))

(load-extension "libguile-xmltokenizer.so" "scm_init_xmltokenizer")

(define (real-pair? x)
  (and (pair? x) (not (list? x))))

(define (tokens->scm tokens)
  (let loop ((result '())
             (tokens tokens))
    (if (null? tokens) result
        (let ((token (car tokens)))
          (cond
           ((string? token) (if (and (pair? result) (string? (car result)))
                                (loop (cons (string-append token (car result)) (cdr result))
                                      (cdr tokens))
                                (loop (cons token result) (cdr tokens))))
           ((pair? token)  (let ((tag (reverse token)))
                             (case (car tag)
                               ((/)
                                (let-values (((element rest) (loop '() (cdr tokens))))
                                  (loop (cons element result) rest)))
                               (else
                                (values `(,(car tag) ,@(plist->alist (cdr tag)) ,@result) (cdr tokens))))))
           (else (error "wrong token in input sequence" token)))))))

(define (file->scm file)
  (call-with-input-file
      file
    (compose tokens->scm xml-tokenize)))

(define (attr->string attrlist)
  (if (null? attrlist)
      ""
      (string-append " "
                     (string-join (map (lambda(x) (format #f "~A=\"~A\"" (car x) (html:quote (cdr x))))
                                       attrlist)
                                  " "))))

(define (sure-list x)
  (if (list? x) x (list x)))

(define (tag? sxml)
  (and (pair? sxml) (symbol? (car sxml))))

(define (comment? sxml)
  (and (pair? sxml) (eq? (car sxml) '!--)))

(define (entity? sxml)
  (and (pair? sxml) (eq? (car sxml) '&)))

(define (script? sxml)
  (and (pair? sxml) (eq? (car sxml) 'script)))

(define (style? sxml)
  (and (pair? sxml) (eq? (car sxml) 'style)))

(define (scm->out sxml write)
  (cond
   ((string? sxml)
    (write (html:quote sxml)))
   ((entity? sxml)
    (write (format #f "&~A;" (cadr sxml))))
   ((comment? sxml)
    (write (format #f "<!--~A-->" (cadr sxml))))
   ((style? sxml)
    (let-values ( ((attr content) (partition real-pair? (cdr sxml)))
                  ((nl) (string #\newline)) )
      (write (format #f "<style~A>~%" (attr->string attr)))
      (for-each (lambda(x) (write x) (write nl)) (list-flat content))
      (write (format #f "</style>~%"))))
   ((script? sxml)
    (let-values ( ((attr content) (partition real-pair? (cdr sxml))) )
      (write (format #f "<script~A>" (attr->string attr)))
      (for-each write (list-flat content)) ;;TODO: use tree-for-each
      (write "</script>")))
   ((tag? sxml)
    (let*-values ( ((tag) (car sxml))
                   ((attr content) (partition real-pair? (cdr sxml))) )
      (write (format #f "<~A~A>" tag (attr->string attr)))
      (scm->out content write)
      (write (format #f "</~A>" tag))))
   (else ;;auto flat list
    (for-each (lambda(x) (scm->out x write)) sxml))))

(define (scm-filter sxml callback)
  (cond
   ((string? sxml)
    sxml)
   ((and (pair? sxml) (symbol? (car sxml)))
    (let*-values ( ((tag) (car sxml))
                   ((attr content) (partition real-pair? (cdr sxml)))
                   ((replacement) (callback tag attr content))
                   ((mode) (and (pair? replacement) (car replacement)))
                   ((new-content) (and (pair? replacement) (cdr replacement))) )
      (case mode
        ((replace-tag)
         new-content)
        (else ;;replace-content,append-content, default
         `(,tag ,@attr
                ,@(if (eq? mode 'replace-content)
                      (list "")
                      (map (lambda (subexp) (scm-filter subexp callback)) content))
                ,@(if (member mode '(replace-content append-content))
                      (sure-list new-content)
                      (list "")))))))
   (else ;;simple sublist
    (map (lambda(item) (scm-filter item callback)) sxml))))

;;;;;;;;;;;;;;;;;;;

(define (optimize-strings lst)
  (let loop ((lst lst)
             (result '()))
    (if (null? lst)
        (reverse result)
        (let-values (( (strings rest) (span string? lst) ))
          (if (null? strings)
              (loop (cdr rest)
                    (cons (car rest) result))
              (loop rest
                    (cons (apply string-append strings) result)))))))

(define (cb-tag cb)
  (vector-ref cb 0))

(define (cb-options cb)
  (vector-ref cb 1))

(define (cb-content cb)
  (vector-ref cb 2))

(define (cb-operation cb)
  (cond-assq 'template-operation (cb-options cb) 'replace-content))

(define (cb-not-operation? x)
  (not (eq? (car x) 'template-operation)))

(define (cb-tag<=? name options)
  (lambda(cb)
    (and (eq? name (cb-tag cb))
         (lset<= equal? (filter cb-not-operation? (cb-options cb)) options))))

(define (run-cb cb options content)
  (map (lambda(item)
         (if (procedure? item)
             (item options content)
             item))
       (cb-content cb)))

(define (make-cb . callbacks)
  (let ((callbacks (filter vector? (list-flat callbacks))))
    (lambda(name options content)
      (let ((cb (find (cb-tag<=? name options) callbacks)))
        (and cb (cons (cb-operation cb) (run-cb cb options content)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;

(define (template filename . callbacks)
  (scm-filter
   (file->scm filename)
   (apply make-cb callbacks)))

(define (tag: name . args)
  (let-values (( (options other) (partition real-pair? args)) )
    (vector (string->symbol name) options (optimize-strings other))))

(define (replace-tag: name . args)
  (apply tag:
         name
         (@ 'template-operation 'replace-tag)
         args))
