(define-module (alterator lookout)
  ;;common modules
  :use-module (alterator object)
  :use-module (alterator algo)
  :use-module (alterator str)
  :use-module (alterator woo)

  ;;simplest message passing
  :use-module (alterator mailbox)
  
  ;;lookout modules
  :use-module (alterator lookout root)
  
  ;;session modules
  :use-module (alterator session common)

  :export (lookout))


(define (auth-permission-granted content)
  (define-operation get-id)
  (define-operation get-user)
  (define-operation resume)
  
  (let ((session (fluid-ref generic-session)))
    `(auth-answer user ,(get-user session)
                  session-id ,(number->string (get-id session))
                  content ,(resume session content))))

;;open a new session on '/std/auth document and pass initial parameters to it
(define (auth-permission-denied params)
  (define-operation start)
  (with-fluids ((generic-session (session-open)))
               (let ((session (fluid-ref generic-session)))
                 (if session
                     (begin
                       (start session
                              (thunk (apply document:popup '/std/auth params) (session-close)))
                       (auth-permission-granted '(alterator-request action "re-get")))
                     '(auth-answer content ((command action "overflow")))))))

;;process incoming lookout message
(define (lookout-request session content params)
  (if session
      (with-fluids ((generic-session session))
                   (auth-permission-granted content))
      (auth-permission-denied params)))

;;process incoming mailbox message
(define (mailbox-request session content)
  (define-operation read-messages)
  (run-mailbox (cdr content))
  `(auth-answer
    content ,(if session (read-messages session) '())))

;;authenticate incoming message
(define (auth-request cmd)
  (let ((content (woo-get-option cmd 'content #f))
	(user	 (woo-get-option cmd 'user #f))
	(session-id (woo-get-option cmd 'session-id #f)))

    (cond ;;switch to session or open a new one for '/std/auth document
      (session-id
       (let ((session (session-find user session-id)))
         (case (cond-car content)
           ((mailbox-request) (mailbox-request session content))
           (else (lookout-request session content (cdr cmd) )))))
      (else (auth-permission-denied (cdr cmd))))))

;main entry point

;;get next command from upper level pass it to engine, process only single command here
(define (lookout-main cmds next)
  (list (auth-request (car cmds))))

(define (make-auto-language next)
  (lambda(cmds)
    (next
     (map (lambda(cmd)
            (cons* (car cmd)
                   'language (session-language)
                   (cdr cmd))) cmds))))

(define (lookout)
  (lambda (cmds next)
    (with-fluids ((woo-gate (make-auto-language next)))
                 (lookout-main cmds next))))

