(define-module (alterator menu)

  :use-module (srfi srfi-1)
  :use-module (srfi srfi-2)
  :use-module (srfi srfi-11)
  :use-module (srfi srfi-13)

  :use-module (alterator str)
  :use-module (alterator algo)
  :use-module (alterator plist)
  :use-module (alterator woo)
  :use-module (alterator gettext)

  :use-module (alterator ensign)

  :use-module (alterator http html)
  :use-module (alterator http template)

  :use-module (alterator configd html)
  :use-module (alterator configd frontend)
  :export (menu))

(define (make-spaces str)
  (let ((str (string-cut-repeated str #\space)))
    (cdr (concatenate (zip
                       (make-list (length str) '(& "nbsp"))
                       str)))))

(define (menu-item url)
  (lambda(item)
    (list  " "
           `(local:a ,(make-spaces (cdr item))
                     ,(@ 'href (string-append "/" (car item)))
                     ,(@ 'class (if (and (string? url) (string=? url (car item)))
                                    "menu-item-selected"
                                    "menu-item"))))))

(define (tag? name)
  (lambda(x)
    (and (pair? x)
         (eq? (car x) name)
	 x)))

(define (same-group? x y)
  (string=? (cddr x) (cddr y)))

(define (main-menu-items all-items)
  (map (lambda(x) (cons* (car x) (cddr x)))
       (delete-duplicates all-items same-group?)))

(define (sub-menu-items group-items)
  (map (lambda(x) (cons* (car x) (cadr x)))
       group-items))

(define (name+title+group x)
  (cons* (woo-get-option x 'name)
         (woo-get-option x 'title)
         (woo-get-option x 'group)))

(define (base-item base-url all-items)
    (find (lambda(x) (string=? (car x) base-url))
           all-items))

(define (need-submenu? base-item base-group)
  (and (pair? base-item)
       (pair? base-group)
       (or (> (length base-group) 1)
           (not (string=? (cddr base-item) (cadr base-item))))))

(define (base-group base-item all-items)
  (and (pair? base-item)
       (let ((group (cddr base-item)))
         (filter (lambda(x) (string=? (cddr x) group))
                 all-items))))

(define (menu-tag url url-args)
  (let* ((all-items (map name+title+group
                         (woo-list "/index"
                                   'language (cond-assoc "language" url-args '("en_US") ))))
         (base-url (base-path url))
         (base-item (base-item base-url all-items))
         (base-group (base-group base-item all-items))
         (top-level-url (and (pair? base-group) (caar base-group))))
    (list
     (tag: "div"
           (@ 'class "menu")
           (map (menu-item top-level-url) (main-menu-items all-items)))
     (replace-tag: "div"
                   (@ 'class "submenu")
                   (if (need-submenu? base-item base-group)
                       (html: "div"
                              (@ 'class "submenu")
                              (map (menu-item base-url) (sub-menu-items base-group)))
                       "")))))

(define (try-unquote scm)
  (if (eq? (car scm) 'quote)
      (cdr scm)
      scm))

(define (help-referrer url)
  (if (string-prefix? "/help" url)
      (substring url (string-length "/help"))
      url))

(define (console-title name _)
  (if (string-null? name)
      std
      (string-append name std)))

(define (help-tag url url-args _)
  (let ((url (if (string=? url "/") "/index" url)))
    (tag: "div"
          (@ 'class "help")
          (if (string-prefix? "/help" url)
              `(local:a ,(@ 'href (help-referrer url)) ,(_ "Return"))
              `(local:a ,(@ 'href (string-append "/help" url)) ,(_ "Help"))))))

(define (title/info url url-args)
  (woo-get-option (woo-read-first (string-append "/index/" (base-path url))
                                  'language (cond-assoc "language" url-args '("en_US")))
                  'title))
        
(define (title/tag tag)
  (and tag (apply string-append (filter string? tag))))

(define (std-title std name)
  (if (string-null? name)
      std
      (string-append name " - " std)))

(define (head-tag url url-args head _)
  (let-values ( ((title rest) (partition (tag? 'title) (cdr head))) )
    (tag: "head"
          (@ 'template-operation 'append-content)
          (html: "title"
                 (or (title/tag (cond-car title))
                     (std-title (_ "ALT Linux Console") (title/info url url-args))))
          rest)))

;;generate main menu
(define (apply-menu  scm design-url url url-args)
  (or (and-let* (((not (eq? (car scm) 'quote)))
                 (html (any (tag? 'html) scm))
                 (head (any (tag? 'head) html))
                 (body (any (tag? 'body) html))
                 (design-url (design-path design-url))
                 (_ (make-translator "alterator-fbi" (cond-assoc "language" url-args '("en_US")))))

                (head-tag url url-args head _)
                
                (template design-url
                          (head-tag url url-args head _)
                          (help-tag url url-args _)
                          (menu-tag (help-referrer url) url-args)
                          (tag: "div"
                                (@ 'class "main")
                                (cdr body))))
      (try-unquote scm)))

;;method, url, args
(define (menu-main design-url cmds next)
  (let*-values (( (method url url-args . rest) (apply values (car cmds))))
    (catch #t
      (thunk (let  ((result (next cmds)))
               (if (or (cond-assoc "async" url-args)
                       (string-ci=? method "woo"))
                   result
                   (apply-menu result design-url url url-args))))
      (html:exception url-args))))

(define (menu design-url)
  (lambda (cmds next)
    (catch #t
      (thunk
       (with-fluids
        ((woo-gate ensign-gate))
        (menu-main design-url cmds next)))
      (html:exception '() ))))
