;description: working with pipes and subprocesses

(define-module (alterator pipe)
	       :use-module  (alterator algo)
               :use-module (alterator terminal)
	       :export (create-process
			 stop-process
			 with-input-from-pipe
			 with-ignored-sigpipe
			 
			 current-ports
			 port-for-read
			 port-for-write))

;we have made special controlling terminal for all childs
;when master process exiting ... all childs will receive SIGHUP signal
(define master+slave (make-pty))


;extended version of ice-9 popen
(define (read-only? x) (eq? x 'read-only))
(define (write-only? x) (eq? x 'write-only))
(define (read-write? x) (eq? x 'read-write))

;create pipe and made nonbuffered write side
(define (create-pipe)
  (let ((p (pipe)))
    (setvbuf (cdr p) _IONBF)
    p))

;return (pid,read-port,write-port)
;TODO: port other code from ice-9 popen
; e.g. guardian's, hooks, etc.
(define (create-process mode prog . args)
  (catch 'system-error
    (thunk
     (let ((read-pipe (and (or (read-only? mode)
                               (read-write? mode))
                           (create-pipe)))
           (write-pipe (and (or (write-only? mode)
                                (read-write? mode))
                            (create-pipe))))
       (let ((pid (primitive-fork)))
         (cond 
          ((zero? pid) ;child work
           
           ;;setup controling terminal
           (if (pair? master+slave)
               (begin (close-fdes (car master+slave))
                      (set-controlling-terminal (cdr master+slave)))
               (format (current-error-port) "warning: unable to setup controlling terminal for backend"))
           
           (if (or (read-only? mode)
                   (read-write? mode))
               (begin (close-port (car read-pipe))
                      (dup2 (fileno (cdr read-pipe)) 1)))
           (if (or (write-only? mode)
                   (read-write? mode))
               (begin (close-port (cdr write-pipe))
                      (dup2 (fileno (car write-pipe)) 0)))
           ;;ports cleanup
           (port-for-each (lambda (pt-entry)
                            (catch #t
                              (thunk
                               (let ((pt-fileno (fileno pt-entry)))
                                 (or (= pt-fileno 0)
                                     (= pt-fileno 1)
                                     (= pt-fileno 2)
                                     (close-fdes pt-fileno))))
                              (lambda(key . args) #f))))
           ;;start subprocess
           (apply execlp prog prog args)
           (throw 'critical-error (format #f "execlp for ~A failed" prog)))
          (else ;parent work
           (list pid
                 (if (or (read-only? mode)
                         (read-write? mode))
                     (begin (close-port (cdr read-pipe))
                            (car read-pipe))
                     #f)
                 (if (or (write-only? mode)
                         (read-write? mode))
                     (begin (close-port (car write-pipe))
                            (cdr write-pipe))
                     #f)))))))
    (lambda args
      (throw 'misc-error
             (cadr args)
             (format #f "~A: ~A" prog (strerror (system-error-errno args)))
	     #f))))

(define (stop-process action p)
  (and (caddr p) (not (port-closed? (caddr p))) (close-port (caddr p)))
  (begin-1
   (catch #t
     (thunk
       (if (eq? action 'terminate) (kill (car p) SIGTERM))
       (status:exit-val (cdr (waitpid (car p)))))
     (lambda (key . args) 0))
   (and (cadr p) (not (port-closed? (cadr p))) (close-port (cadr p)))))
 

(define (with-input-from-pipe cmdline main-thunk)
  (let ((saved-iport #f)
	(process #f))
    (dynamic-wind
      (thunk
	(set! saved-iport (current-input-port))
	(set! process (apply create-process 'read-only (car cmdline) (cdr cmdline)))
	(set-current-input-port (cadr process)))
      main-thunk
      (thunk
	(stop-process 'wait process)
	(set-current-input-port saved-iport)))))

(define (with-ignored-sigpipe main-thunk)
  (let ((saved-action #f))
    (dynamic-wind
      (thunk (set! saved-action (sigaction SIGPIPE SIG_IGN)))
      main-thunk
      (thunk (sigaction SIGPIPE (car saved-action))))))

;all functions works with pair of ports: (for-read,for-write)
(define (current-ports) (cons (current-input-port) (current-output-port)))
(define (port-for-read ports) (cadr ports))
(define (port-for-write ports) (caddr ports))
