(define-module (alterator presentation document)
	       :use-module (srfi srfi-1)

	       :use-module (alterator algo)
	       :use-module (alterator object)

	       :use-module (alterator context)
	       :use-module (alterator atlas)

	       :use-module (alterator security wrap)

               :export (preprocess-file
			preprocess-url))

(define (call-with-void-input proc)
  (let* ((file (%make-void-port "r"))
         (ans (proc file)))
    (close-input-port file)
    ans))

;file mapper item
(define (make-void-item . args)
  (object
    (lambda args
      (apply make-void-item args))
    ((params-of self) '())
    ((call-with-port self proc) (call-with-void-input proc))))


;main engine to lookout input documents

;processing insert instruction
(define (process-insert port instruction initial)
  (append initial
          (preprocess-url (cadr instruction))))
          
;run subroutine to process next environment
(define (process-environment port tag initial)
    (append  initial
	     (list `(,@(cdr tag)
	              ,(apply document->toplevel-begin (read-environment port (cadr tag) '() ))))))

;processing surround instruction
(define (process-surround port instruction initial)
    (append initial
	    ;append result of preprocessing of other file with rest: our rest file
	    (apply preprocess-url
		    (cadr instruction)
		    `((document:quote ,@(read-environment port (gensym) '() ))))))

;analyze pair instruction
(define (process-pair port instruction initial)
  (case (car instruction)
    ((document:insert) (process-insert port instruction initial))
    ((document:envelop) (process-environment port instruction initial))
    ((document:end-envelop) (error "parse-error, unexpected end-envelop" (cadr instruction)))
    ((document:quote) (append initial (cdr instruction)))
    ((document:surround) (process-surround port instruction initial))
    (else (append1 initial instruction))))

;analyze instruction we read
(define (process-instruction port instruction initial)
  (cond 
    ((pair? instruction) (process-pair port instruction initial))
    (else (append1 initial instruction))))

;get next intruction from rest
;demo: return all the rest
(define (get-from-rest port)
  (let ((rest (cadr port)))
    (if (null? rest)
      (read (car port)) ;eof here
      (begin-1 (car rest)
	       (set-car! (cdr port) (cdr rest))))))

;get next instruction from port
(define (get-from-port port)
  (read (car port)))

;get next instruction from input stream (port or already readed data)
;first try to read from file
;then try to return rest
;otherwise return eof
(define (get-instruction port)
  (let ((result (get-from-port port)))
    (if (eof-object? result) ;first try to read from file
      (get-from-rest port)
      result)));return next result from file

;go to the depth of instruction and parse it
(define (parse-in-depth instruction)
  (if (eof-object? instruction)
       instruction
       (cond
	 ((and (pair? instruction)
	       (eq? (car instruction) 'document:quote))
	  instruction)        ;don't parse quotes
	 ((list? instruction)
	  (apply preprocess-file (make-void-item) instruction))
	 (else instruction)))) ; non-lists are not interest for us

;collect data from some environment
;environment ends with end-of-file or at document:ent-evelop instruction
(define (read-environment port tagname initial)
  (let ((instruction (parse-in-depth (get-instruction port))))
    (if (or (eof-object? instruction)
	    (and (pair? instruction)
		 (eq? (car instruction) 'document:end-envelop)
		 (eq? (cadr instruction) tagname)))
	    initial
	    (read-environment port tagname
			     (process-instruction port instruction initial)))))

;preprocess dialog with links and eviroments
(define (preprocess-file file . rest)
  (define-operation call-with-port)
  (call-with-port file
		  (lambda (port)
		    (read-environment (append1 (list port) rest)
				     (gensym) '() ))))

;same as preprocess-file, but also resolving url
(define (preprocess-url  url . rest)
  (let ((path ((from-context 'atlas) url)))
    (or  (and path (apply preprocess-file path rest))
	 '())))

