;user's session in generic look of alterator
(define-module (alterator session common)
  :use-module (srfi srfi-1)

  :use-module (alterator algo)
  :use-module (alterator object)
  :use-module (alterator session state)
  :use-module (alterator session widgets)
  
  :export (generic-session

           session-open
	   session-close
	   session-find
	   session-language
	   
           timer-add
           timer-del

	   without-notifications))


(define generic-session-list '()) ;internal alist of sessions

;TODO: made more strong generator
(define random-state (seed->random-state (current-time)))

(define (make-session-id) (random 100000000 random-state))

(define max-session-count 100)
(define max-session-timeout 10)

(define generic-session (make-fluid))
(fluid-set! generic-session '())

;;live flag is an indicator of unused sessions
(define (make-common-info id)
  (let ((username "anonymous")
        (live #f)
        (timestamp (current-time))
        (language '("en"))) ;english is a default language
    (object
     #f
     ((get-id self) id)
     ((get-timestamp self) timestamp)
     
     ((get-user self) username)
     ((set-user self value) (set! username value))
     ((get-live self) live)
     ((set-live self value) (set! live value))
     ((get-lang self) language)
     ((set-lang self value) (set! language value)))))

(define (make-timers)
  (define-operation append-message-for-id)
  (let ((timer-counter 0))
    (object
     #f
     ;;timer's work
     ((add-timer self)
      (and (zero? timer-counter) (append-message-for-id self ""
                                                '(action "timer"
                                                  value "add")))
      (set! timer-counter (+ 1 timer-counter)))
     ((delete-timer self)
      (and (> timer-counter)
           (begin (set! timer-counter (- 1 timer-counter))
                  (and (zero? timer-counter)
                       (append-message-for-id self ""
                                       '(action "timer"
                                         value "remove")))))))))
     
(define (make-session-object id)
  (define-operation read-messages)
  (define-operation sleep)

    (join
     (make-common-info id)
     (make-session-state)
     (make-widgets-glue)
     (make-timers)
     (object
      #f
      ;;control passing throw continuations
      ((pause self) ;;stop thread until next message arrived
       (sleep self (thunk (read-messages self)))))))

(define (session-alive? x)
        (define-operation get-live)
        (define-operation get-timestamp)

        ;bad session: is dead and with big timeout
         (or (get-live (cdr x))
             (< (- (current-time) (get-timestamp (cdr x)))
                max-session-timeout)))

(define (session-cleanup)
 (set! generic-session-list
       (filter session-alive? generic-session-list)))

         
(define (session-open)
  (session-cleanup)
  ;;TODO: we can allow for some users to override a quote to avoid full DOS
  (and (< (length generic-session-list)
            max-session-count)
      (let* ((id (make-session-id))
             (session (make-session-object id)))
        (set! generic-session-list (alist-set id session generic-session-list))
        session)))

(define (session-close)
  (define-operation get-id)
  (define-operation pause)
  (let ((session (fluid-ref generic-session)))
    (set! generic-session-list (alist-delete (get-id session) generic-session-list))
    (pause session)))

(define (session-find username session-id)
  (define-operation get-user)
  
  (let ((session (cond-cdr (assv (string->number session-id) generic-session-list))))
    (and session
         (string=? (get-user session) username)
         session)))

(define-macro (without-notifications  <instruction> .  <instructions>)
  `(let ()
     (define-operation message-on)
     (define-operation message-off)
     (dynamic-wind
         (lambda() (message-off (fluid-ref generic-session)))
         (lambda() ,<instruction> . ,<instructions>)
         (lambda() (message-on (fluid-ref generic-session))))))


(define (timer-add)
  (define-operation add-timer)
  (add-timer (fluid-ref generic-session)))

(define (timer-del)
  (define-operation delete-timer)
  (delete-timer (fluid-ref generic-session)))


(define (session-language)
  (define-operation get-lang #f)
  (or (get-lang (fluid-ref generic-session)) (list "en")))
					  