(define-module (alterator woo)
  :use-module (srfi srfi-1)
  :use-module (srfi srfi-13)
  :use-module (alterator plist)
  :use-module (alterator algo)
  :use-module (alterator str)
  :export (woo-gate

           woo-get-option
           woo-read-names
           woo-first-command
           woo-error?

           woo
           woo-throw
           woo-try
           woo-try-cmds
           woo-catch

           woo-read
           woo-list
           woo-delete
           woo-new
           woo-write

           woo-read-first
           woo-list-names))

;;set of functions to work with woobus from lookout interfaces or second level backends

;;gateway to next telegraph entry
(define woo-gate (make-fluid))
(fluid-set! woo-gate #f)

;;special version of woo-query
(define (woo-query . cmds)
  (let ((gate (fluid-ref woo-gate)))
    (if (procedure? gate)
        (gate cmds)
        '((/error reason "woo-gate undefined")))))

(define (woo-get-option cmd option . default)
  (cond
   ((and (pair? cmd) (plistq option (cdr cmd))) => cdr)
   (else (if (pair? default) (car default) ""))))

(define (woo-error? cmd)
  (and (pair? cmd)
       (let ((name (car cmd)))
         (and (string? name)
              (or (string-prefix? "/error/" name)
                  (string-prefix? "/error?/" name))))))

;;version with predefinded action field
(define (woo action name . args)
  (woo-query (cons* (if (list? name) name (->string name))
                    'action action
                    args)))

(define (woo-throw . args)
  (throw 'woo-error (apply string-append args)))

;;throw on error
(define (woo-try-cmds cmds)
  (let ((error-answer (find woo-error? cmds)))
    (and error-answer
         (woo-throw (woo-get-option error-answer 'reason)))
    cmds))

;;version with exception support
(define (woo-try action name . args)
  (woo-try-cmds (apply woo action name args)))

;;wrappers over standard actions
(define (woo-read name . args) (apply woo-try "read" name args))
(define (woo-list name . args) (apply woo-try "list" name args))
(define (woo-delete name . args) (apply woo-try "delete" name args))
(define (woo-new name . args) (apply woo-try "new" name args))
(define (woo-write name . args) (apply woo-try "write" name args))

(define (woo-catch run-code catch-code)
  (catch 'woo-error
    run-code
    (lambda (key arg . rest)
      (catch-code arg))))

;;useful helpers
(define (woo-first-command cmdlist)
  (and (pair? cmdlist) (car cmdlist)))

(define (woo-list-names name . args)
  (define (extract-name cmd) (cond-plistq 'name (cdr cmd)))
  (let ((name (->string name)))
    (map extract-name (apply woo-list name args))))

(define (woo-read-first name . args)
  (woo-first-command (apply woo-read name args)))
