#!/usr/bin/guile16 -s
!#

(use-modules (srfi srfi-1)
             (srfi srfi-13)
             (alterator algo)
             (alterator gettext)

             (alterator http cgi)
             (alterator http html)
             (alterator http template)
             (alterator http locale))

;;; localization

(define (cookie-languages)
  (let ((language (assoc "language" (get-cookie))))
    (and language (list (cdr language)))))

(define (make-langlist)
  (map code->locale (or (cookie-languages)
                        (get-languages)
                        '("en-us"))))

(define (cons-language lst)
  (acons "language" (make-langlist) lst))

(define (make-cgi-translator)
  (make-translator "alterator-fbi" (make-langlist)))

;;;

(define (error-message . messages)
  (let ((_ (make-cgi-translator)))
    (template "/var/www/html/fbi/error.html"
              (tag: "h1" (_ "CGI Error"))
              (tag: "div" (@ 'class "error-message")
                   (apply string-append (map _ messages))))))

;;;;;;;;;;;;;;;;;;;;;;;;
;; remote woo calls
(define *socket-name* "/var/run/configd/.sock")

(define (alterator<->cgi url-args)
  (catch 'system-error
    (thunk
     (let ((s (socket PF_UNIX SOCK_STREAM 0)))
       (connect s AF_UNIX *socket-name*)
       (write (list (request-method)
                    (string-append "/" (string-trim-both (path-info) #\/))
                    url-args)
              s)
       (begin-1 (read s)
                (close-port s))))
    (lambda args
      (error-message "Unable to connect to configd service:"
                     (strerror (system-error-errno args))))))
         
;;;;;;;;;;;;;;;;;;;;;;;;
(define (not-href? x)
  (not (eq? (car x) 'href)))

(define (replace-local-refs)
  (replace-tag: "local:a"
                (lambda(options content)
                  `(a ,(@ 'href (string-append (script-name) (cond-assq 'href options "")))
		      ,@(filter not-href? options)
                      ,@content)) ))

(define *xml-entities* '("amp" "apos" "quot" "lt" "gt"))

;;remove submit and reset buttons in async answer
(define (async-optimizations)
  (list
   (replace-tag: "&" ;;remove non-xml entities
                 (lambda(options content)
                   (if (memq (car content) *xml-entities*)
                       `(& ,(car content))
                       "")))
   (replace-tag: "input"
                 (@ 'type "submit") "")
   (replace-tag: "input"
                 (@ 'type "reset") "")))

;;default header parameters
(define (header-defaults scm)
  (scm-filter scm
              (make-cb
               (replace-tag: "head"
                             (lambda (options content)
                               `(head ,@options
                                      ,(html: "script" (@ 'src "/fbi/common.js"))
                                      ,(html: "meta"
                                              (@ 'http-equiv "Cache-Control")
                                              (@ 'value "no-cache"))
                                      ,(html: "meta"
                                              (@ 'http-equiv "Content-Type")
                                              (@ 'value "text/html;charset=utf-8"))
                                      ,@content))))))

(define (template-postprocess scm async)
  (header-defaults
   (scm-filter
    scm
    (make-cb
     (and async (async-optimizations))
     (replace-local-refs)))))

(define (put-template scm async)
  (scm->out (template-postprocess scm async) put-message))

(define (put-doctype)
  (put-message "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0//EN\" \"http://www.w3.org/TR/REC-html40/strict.dtd\">"))

(define (put-header async)
  (put-message (format #f "cache-control: no-cache~%"))
  (if async
      (begin (set-content-type "text/xml")
             (put-message "<?xml version=\"1.0\"?>"))
      (begin (set-content-type "text/html;charset=utf-8")
             (put-doctype)
             (put-message (string #\newline)))))

(define (redirect? answer)
 (and (pair? answer) (eq? (car answer) 'redirect)))

;; request listener
(call-with-content
 (lambda(str)
   (force-output (current-error-port))
   (catch #t
     (thunk
      (or (string? str) (throw 'cgi-error "Bad request"))
      (let* ((url-args (append (decode-content str)
                               (decode-url (query-string))))
             (async (cond-assoc "async" url-args))
             (answer (alterator<->cgi (cons-language url-args))))
        (cond
         ((redirect? answer)
	  (put-message (format #f "Status: 302 Post-redirect-Get~%"))
	  (put-message (format #f "Location:~A~A~%~%" (script-name) (cadr answer)))
	 )
         (else
          (put-header async)
          (put-template answer async)))))
     (lambda (key . args)
       (put-header #f)
       (put-template
          (error-message "Unhandled exception:" (format #f "key=~S,args=~S~%" key args))
	  #f)))))
